<?php
session_start();
include 'config.php';

// Pastikan role adalah anggota
if (!isset($_SESSION['role']) || $_SESSION['role'] != 'Anggota') {
    header("Location: index.php");
    exit;
}

$id_anggota = $_SESSION['id_anggota'];

// Aktifkan error mysqli agar mudah debugging
mysqli_report(MYSQLI_REPORT_ERROR | MYSQLI_REPORT_STRICT);

// Query untuk mendapatkan data anggota
$anggotaQuery = "SELECT nama, nip, jabatan FROM anggota WHERE id_anggota = ?";
$stmt = $conn->prepare($anggotaQuery);
$stmt->bind_param("i", $id_anggota);
$stmt->execute();
$anggotaData = $stmt->get_result()->fetch_assoc();

$message = '';
$message_type = '';

// Proses form pengajuan pinjaman
if ($_SERVER['REQUEST_METHOD'] == 'POST') {
    $jumlah_pinjaman = $_POST['jumlah_pinjaman'];
    $lama_bayar = $_POST['lama_bayar'];
    $tanggal_pengajuan = date('Y-m-d');
    
    // Handle file upload
    $dokumen_pendukung = '';
    if (isset($_FILES['dokumen_pendukung']) && $_FILES['dokumen_pendukung']['error'] == 0) {
        $file_name = $_FILES['dokumen_pendukung']['name'];
        $file_tmp = $_FILES['dokumen_pendukung']['tmp_name'];
        $file_ext = strtolower(pathinfo($file_name, PATHINFO_EXTENSION));
        
        // Validasi ekstensi file
        $allowed_ext = ['pdf', 'doc', 'docx', 'jpg', 'jpeg', 'png'];
        if (in_array($file_ext, $allowed_ext)) {
            // Buat nama file unik
            $new_file_name = $anggotaData['nama'] . '_' . time() . '.' . $file_ext;
            $upload_dir = 'uploads/dokumen_pinjaman/';
            
            // Buat direktori jika belum ada
            if (!file_exists($upload_dir)) {
                mkdir($upload_dir, 0777, true);
            }
            
            $upload_path = $upload_dir . $new_file_name;
            
            if (move_uploaded_file($file_tmp, $upload_path)) {
                $dokumen_pendukung = $new_file_name;
            } else {
                $message = 'Gagal mengupload dokumen pendukung.';
                $message_type = 'danger';
            }
        } else {
            $message = 'Format file tidak diizinkan. Gunakan PDF, DOC, DOCX, JPG, JPEG, atau PNG.';
            $message_type = 'danger';
        }
    }
    
    // Insert data pinjaman jika tidak ada error
    if (empty($message)) {
        try {
            $insertQuery = "INSERT INTO pinjaman (id_anggota, jumlah_pinjaman, lama_bayar, status, tanggal_pengajuan, dokumen_pendukung) VALUES (?, ?, ?, 'pending', ?, ?)";
            $stmt = $conn->prepare($insertQuery);
            $stmt->bind_param("idiss", $id_anggota, $jumlah_pinjaman, $lama_bayar, $tanggal_pengajuan, $dokumen_pendukung);
            
            if ($stmt->execute()) {
                $message = 'Pengajuan pinjaman berhasil disubmit! Silakan tunggu konfirmasi dari admin.';
                $message_type = 'success';
            } else {
                $message = 'Gagal mengajukan pinjaman. Silakan coba lagi.';
                $message_type = 'danger';
            }
        } catch (Exception $e) {
            $message = 'Terjadi kesalahan: ' . $e->getMessage();
            $message_type = 'danger';
        }
    }
}
?>

<!DOCTYPE html>
<html lang="id">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Ajukan Pinjaman - Koperasi Pusdatin</title>
    <script src="https://cdn.tailwindcss.com"></script>
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.0.0/css/all.min.css">
    <script>
        tailwind.config = {
            theme: {
                extend: {
                    colors: {
                        primary: '#667eea',
                        secondary: '#764ba2',
                    }
                }
            }
        }
    </script>
    <style>
        @import url('https://fonts.googleapis.com/css2?family=Inter:wght@300;400;500;600;700;800&display=swap');
        body {
            font-family: 'Inter', sans-serif;
        }
    </style>
</head>
<body class="bg-gradient-to-br from-purple-50 to-blue-50">
    
    <!-- Sidebar -->
    <aside id="sidebar" class="fixed top-0 left-0 z-40 w-64 h-screen transition-transform -translate-x-full sm:translate-x-0">
        <div class="h-full px-3 py-4 overflow-y-auto bg-gradient-to-b from-purple-600 to-indigo-700 shadow-2xl">
            <!-- Logo Section -->
            <div class="flex flex-col items-center mb-8 pb-6 border-b border-white/20">
                <div class="w-24 h-24 bg-white rounded-full flex items-center justify-center mb-3 shadow-lg">
                    <img src="img/logo.png" alt="Logo Koperasi" class="w-20 h-20 object-contain" onerror="this.style.display='none'; this.nextElementSibling.style.display='block';">
                    <i class="fas fa-university text-purple-600 text-4xl hidden"></i>
                </div>
                <h2 class="text-white font-bold text-lg text-center">Koperasi Pusdatin</h2>
                <p class="text-purple-200 text-xs mt-1">Sistem Informasi Koperasi</p>
            </div>

            <!-- Profile Section -->
            <div class="mb-6 p-4 bg-white/10 rounded-lg backdrop-blur-sm">
                <div class="flex items-center space-x-3">
                    <div class="w-12 h-12 bg-white rounded-full flex items-center justify-center">
                        <i class="fas fa-user text-purple-600 text-xl"></i>
                    </div>
                    <div class="flex-1 min-w-0">
                        <p class="text-white font-semibold text-sm truncate"><?= htmlspecialchars($anggotaData['nama'] ?? 'Anggota') ?></p>
                        <p class="text-purple-200 text-xs truncate"><?= htmlspecialchars($anggotaData['nip'] ?? '-') ?></p>
                    </div>
                </div>
            </div>

            <!-- Navigation Menu -->
            <nav class="space-y-2">
                <a href="dashboard_anggota.php" class="flex items-center p-3 text-white rounded-lg hover:bg-white/20 transition-all duration-200 group">
                    <i class="fas fa-home w-6 text-center group-hover:scale-110 transition-transform"></i>
                    <span class="ml-3 font-medium">Dashboard</span>
                </a>
                
                <a href="ajukan_pinjaman.php" class="flex items-center p-3 text-white bg-white/20 rounded-lg hover:bg-white/30 transition-all duration-200 group">
                    <i class="fas fa-money-bill-wave w-6 text-center group-hover:scale-110 transition-transform"></i>
                    <span class="ml-3 font-medium">Ajukan Pinjaman</span>
                </a>
                
                <a href="status_pinjaman.php" class="flex items-center p-3 text-white rounded-lg hover:bg-white/20 transition-all duration-200 group">
                    <i class="fas fa-file-contract w-6 text-center group-hover:scale-110 transition-transform"></i>
                    <span class="ml-3 font-medium">Status Pinjaman</span>
                </a>
                
                <a href="lihat_transaksi_defmart.php" class="flex items-center p-3 text-white rounded-lg hover:bg-white/20 transition-all duration-200 group">
                    <i class="fas fa-shopping-cart w-6 text-center group-hover:scale-110 transition-transform"></i>
                    <span class="ml-3 font-medium">Transaksi Defmart</span>
                </a>
                
                <a href="lihat_shu.php" class="flex items-center p-3 text-white rounded-lg hover:bg-white/20 transition-all duration-200 group">
                    <i class="fas fa-chart-line w-6 text-center group-hover:scale-110 transition-transform"></i>
                    <span class="ml-3 font-medium">Pembagian SHU</span>
                </a>

                <div class="pt-4 mt-4 border-t border-white/20">
                    <a href="logout.php" class="flex items-center p-3 text-white rounded-lg hover:bg-red-500/30 transition-all duration-200 group">
                        <i class="fas fa-sign-out-alt w-6 text-center group-hover:scale-110 transition-transform"></i>
                        <span class="ml-3 font-medium">Logout</span>
                    </a>
                </div>
            </nav>
        </div>
    </aside>

    <!-- Main Content -->
    <div class="sm:ml-64">
        <!-- Header -->
        <header class="bg-white shadow-md sticky top-0 z-30">
            <div class="flex items-center justify-between px-6 py-4">
                <div class="flex items-center">
                    <button id="toggleSidebar" class="text-gray-600 hover:text-purple-600 focus:outline-none sm:hidden">
                        <i class="fas fa-bars text-2xl"></i>
                    </button>
                    <h1 class="text-2xl font-bold text-gray-800 ml-4">
                        <i class="fas fa-money-bill-wave text-purple-600 mr-2"></i>
                        Ajukan Pinjaman
                    </h1>
                </div>
                <div class="flex items-center space-x-4">
                    <span class="text-gray-600 text-sm">
                        <i class="fas fa-calendar-alt mr-2"></i>
                        <?= date('l, d F Y') ?>
                    </span>
                </div>
            </div>
        </header>

        <!-- Content Area -->
        <main class="p-6">
            <div class="max-w-4xl mx-auto">
                <!-- Info Card -->
                <div class="bg-gradient-to-r from-blue-500 to-cyan-500 rounded-2xl shadow-lg p-6 mb-6 text-white">
                    <div class="flex items-center justify-between">
                        <div class="flex-1">
                            <h2 class="text-2xl font-bold mb-2">
                                <i class="fas fa-file-contract mr-2"></i>
                                Pengajuan Pinjaman
                            </h2>
                            <p class="text-blue-100">Lengkapi form berikut untuk mengajukan pinjaman ke koperasi</p>
                        </div>
                        <div class="hidden md:block">
                            <i class="fas fa-money-check-alt text-white/30 text-7xl"></i>
                        </div>
                    </div>
                </div>

                <!-- Alert Messages -->
                <?php if (!empty($message)): ?>
                <div class="mb-6 p-4 rounded-lg <?= $message_type == 'success' ? 'bg-green-100 border-l-4 border-green-500 text-green-700' : 'bg-red-100 border-l-4 border-red-500 text-red-700' ?>">
                    <div class="flex items-center">
                        <i class="fas fa-<?= $message_type == 'success' ? 'check-circle' : 'exclamation-circle' ?> mr-3 text-2xl"></i>
                        <p class="font-medium"><?= $message ?></p>
                    </div>
                </div>
                <?php endif; ?>

                <!-- Form Card -->
                <div class="bg-white rounded-2xl shadow-xl overflow-hidden">
                    <div class="bg-gradient-to-r from-purple-600 to-indigo-600 px-6 py-4">
                        <h3 class="text-xl font-bold text-white">
                            <i class="fas fa-edit mr-2"></i>
                            Form Pengajuan Pinjaman
                        </h3>
                    </div>

                    <div class="p-6">
                        <!-- Data Anggota -->
                        <div class="mb-6">
                            <h4 class="text-lg font-semibold text-gray-700 mb-3">
                                <i class="fas fa-user mr-2 text-purple-600"></i>
                                Data Pemohon
                            </h4>
                            <div class="bg-gray-50 rounded-lg p-4 grid grid-cols-1 md:grid-cols-3 gap-4">
                                <div>
                                    <p class="text-sm text-gray-600 font-semibold">Nama</p>
                                    <p class="text-gray-800"><?= htmlspecialchars($anggotaData['nama'] ?? '-') ?></p>
                                </div>
                                <div>
                                    <p class="text-sm text-gray-600 font-semibold">NIP</p>
                                    <p class="text-gray-800"><?= htmlspecialchars($anggotaData['nip'] ?? '-') ?></p>
                                </div>
                                <div>
                                    <p class="text-sm text-gray-600 font-semibold">Jabatan</p>
                                    <p class="text-gray-800"><?= htmlspecialchars($anggotaData['jabatan'] ?? '-') ?></p>
                                </div>
                            </div>
                        </div>

                        <!-- Form -->
                        <form method="POST" enctype="multipart/form-data" class="space-y-6">
                            <div class="grid grid-cols-1 md:grid-cols-2 gap-6">
                                <!-- Jumlah Pinjaman -->
                                <div>
                                    <label for="jumlah_pinjaman" class="block text-sm font-semibold text-gray-700 mb-2">
                                        <i class="fas fa-money-bill text-green-600 mr-2"></i>
                                        Jumlah Pinjaman <span class="text-red-500">*</span>
                                    </label>
                                    <div class="relative">
                                        <span class="absolute left-3 top-1/2 -translate-y-1/2 text-gray-500 font-semibold">Rp</span>
                                        <input type="number" id="jumlah_pinjaman" name="jumlah_pinjaman" 
                                               class="w-full pl-12 pr-4 py-3 border border-gray-300 rounded-lg focus:ring-2 focus:ring-purple-500 focus:border-transparent"
                                               min="100000" max="100000000" step="1000" required>
                                    </div>
                                    <p class="text-xs text-gray-500 mt-1">Minimal Rp 100.000, maksimal Rp 100.000.000</p>
                                </div>

                                <!-- Lama Bayar -->
                                <div>
                                    <label for="lama_bayar" class="block text-sm font-semibold text-gray-700 mb-2">
                                        <i class="fas fa-calendar-alt text-blue-600 mr-2"></i>
                                        Jangka Waktu Pembayaran <span class="text-red-500">*</span>
                                    </label>
                                    <select id="lama_bayar" name="lama_bayar" 
                                            class="w-full px-4 py-3 border border-gray-300 rounded-lg focus:ring-2 focus:ring-purple-500 focus:border-transparent" required>
                                        <option value="">Pilih jangka waktu...</option>
                                        <option value="6">6 Bulan</option>
                                        <option value="12">12 Bulan (1 Tahun)</option>
                                        <option value="18">18 Bulan</option>
                                        <option value="24">24 Bulan (2 Tahun)</option>
                                        <option value="36">36 Bulan (3 Tahun)</option>
                                        <option value="48">48 Bulan (4 Tahun)</option>
                                        <option value="60">60 Bulan (5 Tahun)</option>
                                    </select>
                                </div>
                            </div>

                            <!-- Dokumen Pendukung -->
                            <div>
                                <label for="dokumen_pendukung" class="block text-sm font-semibold text-gray-700 mb-2">
                                    <i class="fas fa-paperclip text-yellow-600 mr-2"></i>
                                    Dokumen Pendukung
                                </label>
                                <input type="file" id="dokumen_pendukung" name="dokumen_pendukung" 
                                       class="w-full px-4 py-3 border border-gray-300 rounded-lg focus:ring-2 focus:ring-purple-500 focus:border-transparent"
                                       accept=".pdf,.doc,.docx,.jpg,.jpeg,.png">
                                <p class="text-xs text-gray-500 mt-1">
                                    Upload dokumen pendukung (SK, slip gaji, dll). Format: PDF, DOC, DOCX, JPG, PNG. Maksimal 5MB.
                                </p>
                            </div>

                            <!-- Info Box -->
                            <div class="bg-blue-50 border-l-4 border-blue-500 p-4 rounded-r-lg">
                                <h5 class="font-semibold text-blue-800 mb-2">
                                    <i class="fas fa-info-circle mr-2"></i>
                                    Informasi Penting:
                                </h5>
                                <ul class="text-sm text-blue-700 space-y-1 ml-4">
                                    <li>• Bunga pinjaman akan dihitung sesuai kebijakan koperasi</li>
                                    <li>• Persetujuan pinjaman tergantung pada hasil verifikasi dokumen</li>
                                    <li>• Proses persetujuan membutuhkan waktu 3-7 hari kerja</li>
                                    <li>• Pastikan semua data yang diisi sudah benar</li>
                                </ul>
                            </div>

                            <!-- Submit Button -->
                            <div class="flex justify-center pt-4">
                                <button type="submit" class="px-8 py-3 bg-gradient-to-r from-purple-600 to-indigo-600 text-white font-semibold rounded-lg shadow-lg hover:shadow-xl transform hover:-translate-y-1 transition-all duration-200">
                                    <i class="fas fa-paper-plane mr-2"></i>
                                    Ajukan Pinjaman
                                </button>
                            </div>
                        </form>
                    </div>
                </div>
            </div>
        </main>
    </div>

    <script>
        // Toggle Sidebar untuk mobile
        const toggleSidebar = document.getElementById('toggleSidebar');
        const sidebar = document.getElementById('sidebar');

        if (toggleSidebar) {
            toggleSidebar.addEventListener('click', () => {
                sidebar.classList.toggle('-translate-x-full');
            });
        }

        // Format input rupiah
        document.getElementById('jumlah_pinjaman').addEventListener('input', function(e) {
            let value = this.value.replace(/[^\d]/g, '');
            this.value = value;
        });

        // Preview perhitungan cicilan
        document.getElementById('jumlah_pinjaman').addEventListener('input', calculateLoan);
        document.getElementById('lama_bayar').addEventListener('change', calculateLoan);

        function calculateLoan() {
            const jumlah = document.getElementById('jumlah_pinjaman').value;
            const lama = document.getElementById('lama_bayar').value;
            
            if (jumlah && lama) {
                const bunga = 0.12;
                const bungaBulanan = bunga / 12;
                const jumlahBulan = parseInt(lama);
                
                const cicilan = (jumlah * bungaBulanan * Math.pow(1 + bungaBulanan, jumlahBulan)) / 
                               (Math.pow(1 + bungaBulanan, jumlahBulan) - 1);
                
                console.log('Estimasi cicilan per bulan: Rp ' + Math.round(cicilan).toLocaleString('id-ID'));
            }
        }
    </script>
</body>
</html>