<?php
session_start();
include 'config.php';

// Cek apakah user sudah login dan memiliki role Kasir Defmart
if (!isset($_SESSION['role']) || $_SESSION['role'] !== 'Kasir Defmart') {
    header("Location: index.php");
    exit;
}

$id_user = $_SESSION['id_user'];
$username = $_SESSION['username'];

// Filter
$filter_bulan = isset($_GET['filter_bulan']) ? $_GET['filter_bulan'] : date('Y-m');

// Ambil barang paling sering laku
$query_barang = "SELECT nama_barang, 
                 COUNT(*) as jumlah_terjual, 
                 SUM(qty) as total_qty, 
                 SUM(total_harga) as total_nilai,
                 (SUM(qty) + (COUNT(*) * 10)) as skor_popularitas
                 FROM defmart_transaksi
                 WHERE DATE_FORMAT(tanggal_transaksi, '%Y-%m') = ?
                 GROUP BY nama_barang
                 ORDER BY skor_popularitas DESC, total_qty DESC, jumlah_terjual DESC
                 LIMIT 10";

$stmt_barang = $conn->prepare($query_barang);
if (!$stmt_barang) {
    die("Prepare statement gagal: " . $conn->error);
}
$stmt_barang->bind_param("s", $filter_bulan);
$stmt_barang->execute();
$barang_result = $stmt_barang->get_result();
?>

<!DOCTYPE html>
<html lang="id">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Barang Terlaris - Koperasi PUSDATIN</title>
    <script src="https://cdn.tailwindcss.com"></script>
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.0.0/css/all.min.css">
    <style>
        .gradient-primary { background: linear-gradient(135deg, #667eea 0%, #764ba2 100%); }
    </style>
</head>
<body class="bg-gray-100">

<!-- Sidebar -->
<?php include 'sidebar_kasir.php'; ?>

<!-- Main Content -->
<div class="ml-64 min-h-screen">
    <!-- Top Bar -->
    <div class="bg-white shadow-sm sticky top-0 z-30">
        <div class="px-6 py-4">
            <h1 class="text-2xl font-bold text-gray-800">
                <i class="fas fa-fire mr-2 text-red-500"></i>Barang Terlaris
            </h1>
        </div>
    </div>

    <!-- Content Area -->
    <div class="p-6">
        <!-- Filter Section -->
        <div class="bg-white rounded-xl shadow-sm p-6 mb-6">
            <h3 class="text-lg font-semibold text-gray-800 mb-4">Periode</h3>
            <div class="flex gap-4 items-end">
                <div class="flex-1">
                    <label class="block text-sm font-medium text-gray-700 mb-2">Pilih Bulan</label>
                    <input type="month" id="filter_bulan" class="w-full px-4 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-transparent" value="<?= $filter_bulan ?>">
                </div>
                <button onclick="applyFilter()" class="px-6 py-2 gradient-primary text-white font-semibold rounded-lg hover:shadow-lg transition">
                    <i class="fas fa-filter mr-2"></i>Terapkan
                </button>
            </div>
        </div>

        <!-- Tabel Barang Terlaris -->
        <div class="bg-white rounded-xl shadow-sm overflow-hidden">
            <div class="p-6 border-b border-gray-200">
                <h3 class="text-lg font-semibold text-gray-800">Top 10 Barang Terlaris</h3>
                <p class="text-sm text-gray-600 mt-1">Diurutkan berdasarkan kombinasi total qty dan frekuensi pembelian</p>
            </div>
            <div class="overflow-x-auto">
                <table class="w-full">
                    <thead>
                        <tr class="bg-gray-50 border-b border-gray-200">
                            <th class="px-6 py-3 text-left text-sm font-semibold text-gray-700">Ranking</th>
                            <th class="px-6 py-3 text-left text-sm font-semibold text-gray-700">Nama Barang</th>
                            <th class="px-6 py-3 text-left text-sm font-semibold text-gray-700">Frekuensi</th>
                            <th class="px-6 py-3 text-left text-sm font-semibold text-gray-700">Total Qty</th>
                            <th class="px-6 py-3 text-left text-sm font-semibold text-gray-700">Total Nilai</th>
                            <th class="px-6 py-3 text-left text-sm font-semibold text-gray-700">% dari Total</th>
                        </tr>
                    </thead>
                    <tbody class="divide-y divide-gray-200">
                        <?php 
                        $rank = 1;
                        $total_nilai_all = 0;
                        $barang_data = [];
                        while ($row = $barang_result->fetch_assoc()) {
                            $barang_data[] = $row;
                            $total_nilai_all += $row['total_nilai'];
                        }
                        
                        foreach ($barang_data as $row): 
                            $persentase = $total_nilai_all > 0 ? ($row['total_nilai'] / $total_nilai_all) * 100 : 0;
                            
                            // Badge color based on ranking
                            $badge_color = 'bg-yellow-100 text-yellow-700';
                            if ($rank == 1) $badge_color = 'bg-yellow-400 text-white';
                            else if ($rank == 2) $badge_color = 'bg-gray-300 text-gray-700';
                            else if ($rank == 3) $badge_color = 'bg-orange-300 text-orange-800';
                        ?>
                            <tr class="hover:bg-gray-50 transition">
                                <td class="px-6 py-4 text-sm">
                                    <span class="px-3 py-1 rounded-full font-bold <?= $badge_color ?>">
                                        #<?= $rank++ ?>
                                    </span>
                                </td>
                                <td class="px-6 py-4 text-sm font-semibold text-gray-900"><?= htmlspecialchars($row['nama_barang']) ?></td>
                                <td class="px-6 py-4 text-sm">
                                    <span class="px-3 py-1 bg-blue-100 text-blue-700 rounded-full text-xs font-semibold">
                                        <?= $row['jumlah_terjual'] ?>x
                                    </span>
                                </td>
                                <td class="px-6 py-4 text-sm">
                                    <span class="px-3 py-1 bg-green-100 text-green-700 rounded-full text-xs font-semibold">
                                        <?= $row['total_qty'] ?> pcs
                                    </span>
                                </td>
                                <td class="px-6 py-4 text-sm font-semibold text-gray-900">Rp <?= number_format($row['total_nilai'], 0, ',', '.') ?></td>
                                <td class="px-6 py-4 text-sm">
                                    <div class="flex items-center space-x-2">
                                        <div class="flex-1 bg-gray-200 rounded-full h-2">
                                            <div class="h-2 rounded-full gradient-primary" style="width: <?= $persentase ?>%"></div>
                                        </div>
                                        <span class="text-xs font-semibold text-gray-600"><?= number_format($persentase, 1) ?>%</span>
                                    </div>
                                </td>
                            </tr>
                        <?php endforeach; ?>
                    </tbody>
                </table>
                <?php if (count($barang_data) == 0): ?>
                    <div class="p-6 text-center text-gray-500">
                        <i class="fas fa-inbox text-4xl mb-3 block text-gray-300"></i>
                        Tidak ada data barang untuk periode ini
                    </div>
                <?php endif; ?>
            </div>
        </div>
    </div>
</div>

<script>
    function applyFilter() {
        const bulan = document.getElementById('filter_bulan').value;
        window.location.href = `?filter_bulan=${bulan}`;
    }
</script>

</body>
</html>