<?php
session_start();
include 'config.php';

// Cek apakah user sudah login dan memiliki role Payroll/Juru Bayar
if (!isset($_SESSION['role']) || $_SESSION['role'] !== 'Payroll') {
    header("Location: index.php");
    exit;
}

$username = $_SESSION['username'];
$id_user = $_SESSION['id_user'] ?? 0;
$sukses = $error = '';

// Get current page from URL
$current_page = isset($_GET['page']) ? $_GET['page'] : 'dashboard';

// Proses Approve Pinjaman oleh Payroll
if (isset($_POST['approve_payroll'])) {
    $id_pinjaman = intval($_POST['id_pinjaman']);
    $catatan = trim($_POST['catatan_payroll']);
    $tanggal_approve = date('Y-m-d H:i:s');
    
    $stmt = $conn->prepare("UPDATE pinjaman SET status = 'disetujui', approved_by_payroll = ?, tanggal_approve_payroll = ?, catatan_payroll = ? WHERE id_pinjaman = ? AND approved_by_admin IS NOT NULL");
    $stmt->bind_param("issi", $id_user, $tanggal_approve, $catatan, $id_pinjaman);
    
    if ($stmt->execute()) {
        header("Location: " . $_SERVER['PHP_SELF'] . "?page=approval_pinjaman&success=1");
        exit;
    } else {
        $error = "Gagal menyetujui pinjaman!";
    }
}

// Proses Reject Pinjaman oleh Payroll
if (isset($_POST['reject_payroll'])) {
    $id_pinjaman = intval($_POST['id_pinjaman']);
    $catatan = trim($_POST['catatan_payroll']);
    $tanggal_approve = date('Y-m-d H:i:s');
    
    $stmt = $conn->prepare("UPDATE pinjaman SET status = 'ditolak', approved_by_payroll = ?, tanggal_approve_payroll = ?, catatan_payroll = ? WHERE id_pinjaman = ?");
    $stmt->bind_param("issi", $id_user, $tanggal_approve, $catatan, $id_pinjaman);
    
    if ($stmt->execute()) {
        header("Location: " . $_SERVER['PHP_SELF'] . "?page=approval_pinjaman&success=2");
        exit;
    } else {
        $error = "Gagal menolak pinjaman!";
    }
}

// Handle success messages
if (isset($_GET['success'])) {
    switch ($_GET['success']) {
        case 1: $sukses = "Pinjaman berhasil disetujui!"; break;
        case 2: $sukses = "Pinjaman berhasil ditolak!"; break;
    }
}

// Filter untuk approval pinjaman
$filter_status = isset($_GET['status']) ? $_GET['status'] : 'waiting';
$filter_search = isset($_GET['search']) ? trim($_GET['search']) : '';

$where = [];
if ($filter_status == 'waiting') {
    $where[] = "p.approved_by_admin IS NOT NULL AND p.approved_by_payroll IS NULL AND p.status != 'ditolak'";
} elseif ($filter_status == 'approved') {
    $where[] = "p.approved_by_payroll IS NOT NULL AND p.status = 'disetujui'";
} elseif ($filter_status == 'rejected') {
    $where[] = "p.status = 'ditolak'";
}

if ($filter_search) {
    $where[] = "(a.nama LIKE '%$filter_search%' OR a.nip LIKE '%$filter_search%')";
}

$where_clause = count($where) > 0 ? "WHERE " . implode(" AND ", $where) : "";

// Ambil data pinjaman
$pinjaman = $conn->query("
    SELECT 
        p.*,
        a.nama,
        a.nip,
        admin.username as admin_name,
        payroll.username as payroll_name
    FROM pinjaman p
    JOIN anggota a ON p.id_anggota = a.id_anggota
    LEFT JOIN user admin ON p.approved_by_admin = admin.id_user
    LEFT JOIN user payroll ON p.approved_by_payroll = payroll.id_user
    $where_clause
    ORDER BY p.tanggal_approve_admin DESC, p.id_pinjaman DESC
");

// Statistik Dashboard
$stats_waiting = $conn->query("SELECT COUNT(*) as total FROM pinjaman WHERE approved_by_admin IS NOT NULL AND approved_by_payroll IS NULL AND status != 'ditolak'")->fetch_assoc();
$stats_approved = $conn->query("SELECT COUNT(*) as total FROM pinjaman WHERE approved_by_payroll IS NOT NULL AND status = 'disetujui'")->fetch_assoc();
$stats_rejected = $conn->query("SELECT COUNT(*) as total FROM pinjaman WHERE approved_by_payroll IS NOT NULL AND status = 'ditolak'")->fetch_assoc();
$stats_nilai = $conn->query("SELECT SUM(jumlah_pinjaman) as total FROM pinjaman WHERE approved_by_payroll IS NOT NULL AND status = 'disetujui'")->fetch_assoc();

// Ganti bagian statistik Bon Defmart (sekitar baris 94-95)
// DARI:
$stats_bon = $conn->query("SELECT COUNT(DISTINCT id_anggota) as total_anggota, SUM(total_harga) as total_bon FROM defmart_transaksi WHERE status_bayar = 'belum_lunas'")->fetch_assoc();

// MENJADI:
$stats_bon = $conn->query("SELECT COUNT(DISTINCT id_anggota) as total_anggota, SUM(total_harga) as total_bon FROM defmart_transaksi WHERE metode_pembayaran = 'Bon'")->fetch_assoc();
$data_pinjaman = [];
while ($row = $pinjaman->fetch_assoc()) {
    $data_pinjaman[] = $row;
}
?>

<!DOCTYPE html>
<html lang="id">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Dashboard Payroll - Koperasi PUSDATIN</title>
    
    <!-- Bootstrap CSS -->
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/css/bootstrap.min.css" rel="stylesheet">
    
    <!-- Tailwind CSS -->
    <script src="https://cdn.tailwindcss.com"></script>
    
    <!-- Font Awesome -->
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.4.0/css/all.min.css">
    
    <!-- Google Fonts -->
    <link href="https://fonts.googleapis.com/css2?family=Inter:wght@300;400;500;600;700;800;900&display=swap" rel="stylesheet">
    
    <style>
        * {
            font-family: 'Inter', sans-serif;
        }

        :root {
            --sidebar-width: 280px;
            --primary-color: #6366f1;
            --primary-dark: #4f46e5;
            --secondary-color: #8b5cf6;
            --success-color: #10b981;
            --warning-color: #f59e0b;
            --danger-color: #ef4444;
            --info-color: #3b82f6;
        }

        body {
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            min-height: 100vh;
        }

        /* Sidebar Styles */
        .sidebar {
            position: fixed;
            top: 0;
            left: 0;
            height: 100vh;
            width: var(--sidebar-width);
            background: linear-gradient(180deg, #1e293b 0%, #0f172a 100%);
            box-shadow: 4px 0 20px rgba(0, 0, 0, 0.1);
            z-index: 1000;
            overflow-y: auto;
            transition: all 0.3s ease;
        }

        .sidebar::-webkit-scrollbar {
            width: 6px;
        }

        .sidebar::-webkit-scrollbar-track {
            background: rgba(255, 255, 255, 0.05);
        }

        .sidebar::-webkit-scrollbar-thumb {
            background: rgba(255, 255, 255, 0.2);
            border-radius: 10px;
        }

        .logo-container {
            padding: 2rem 1.5rem;
            border-bottom: 1px solid rgba(255, 255, 255, 0.1);
            text-align: center;
        }

        .logo-placeholder {
            width: 80px;
            height: 80px;
            background: linear-gradient(135deg, var(--primary-color), var(--secondary-color));
            border-radius: 20px;
            display: flex;
            align-items: center;
            justify-content: center;
            margin: 0 auto 1rem;
            box-shadow: 0 8px 20px rgba(99, 102, 241, 0.3);
        }

        .logo-placeholder i {
            font-size: 2.5rem;
            color: white;
        }

        .sidebar-menu {
            padding: 1.5rem 0;
        }

        .menu-item {
            padding: 0.875rem 1.5rem;
            color: rgba(255, 255, 255, 0.7);
            text-decoration: none;
            display: flex;
            align-items: center;
            transition: all 0.3s ease;
            margin: 0.25rem 0.75rem;
            border-radius: 12px;
            font-weight: 500;
        }

        .menu-item i {
            margin-right: 1rem;
            font-size: 1.25rem;
            width: 24px;
            text-align: center;
        }

        .menu-item:hover {
            background: rgba(255, 255, 255, 0.1);
            color: white;
            transform: translateX(5px);
        }

        .menu-item.active {
            background: linear-gradient(135deg, var(--primary-color), var(--secondary-color));
            color: white;
            box-shadow: 0 4px 15px rgba(99, 102, 241, 0.4);
        }

        .menu-section-title {
            padding: 1.5rem 1.5rem 0.5rem;
            color: rgba(255, 255, 255, 0.5);
            font-size: 0.75rem;
            font-weight: 700;
            text-transform: uppercase;
            letter-spacing: 0.05em;
        }

        /* Main Content */
        .main-content {
            margin-left: var(--sidebar-width);
            padding: 2rem;
            min-height: 100vh;
        }

        /* Top Bar */
        .top-bar {
            background: rgba(255, 255, 255, 0.95);
            backdrop-filter: blur(10px);
            border-radius: 20px;
            padding: 1.5rem 2rem;
            margin-bottom: 2rem;
            box-shadow: 0 4px 20px rgba(0, 0, 0, 0.1);
            display: flex;
            justify-content: space-between;
            align-items: center;
        }

        /* Cards */
        .stat-card {
            background: white;
            border-radius: 20px;
            padding: 1.75rem;
            box-shadow: 0 4px 20px rgba(0, 0, 0, 0.08);
            transition: all 0.3s ease;
            border: 2px solid transparent;
            height: 100%;
        }

        .stat-card:hover {
            transform: translateY(-8px);
            box-shadow: 0 12px 40px rgba(0, 0, 0, 0.15);
        }

        .stat-card-icon {
            width: 64px;
            height: 64px;
            border-radius: 16px;
            display: flex;
            align-items: center;
            justify-content: center;
            margin-bottom: 1rem;
        }

        .stat-card-icon i {
            font-size: 1.75rem;
            color: white;
        }

        .content-card {
            background: white;
            border-radius: 20px;
            padding: 2rem;
            box-shadow: 0 4px 20px rgba(0, 0, 0, 0.08);
            margin-bottom: 2rem;
        }

        .loan-card {
            background: white;
            border-radius: 20px;
            overflow: hidden;
            box-shadow: 0 4px 20px rgba(0, 0, 0, 0.08);
            transition: all 0.3s ease;
            margin-bottom: 1.5rem;
        }

        .loan-card:hover {
            transform: translateY(-5px);
            box-shadow: 0 12px 40px rgba(0, 0, 0, 0.15);
        }

        /* Buttons */
        .btn-custom {
            padding: 0.75rem 1.5rem;
            border-radius: 12px;
            font-weight: 600;
            transition: all 0.3s ease;
            border: none;
        }

        .btn-custom:hover {
            transform: translateY(-2px);
            box-shadow: 0 8px 20px rgba(0, 0, 0, 0.2);
        }

        .btn-primary-custom {
            background: linear-gradient(135deg, var(--primary-color), var(--secondary-color));
            color: white;
        }

        .btn-success-custom {
            background: linear-gradient(135deg, #10b981, #059669);
            color: white;
        }

        .btn-danger-custom {
            background: linear-gradient(135deg, #ef4444, #dc2626);
            color: white;
        }

        .btn-info-custom {
            background: linear-gradient(135deg, #3b82f6, #2563eb);
            color: white;
        }

        /* Badge */
        .badge-custom {
            padding: 0.5rem 1rem;
            border-radius: 10px;
            font-weight: 600;
            font-size: 0.875rem;
        }

        /* Modal */
        .modal-content {
            border-radius: 20px;
            border: none;
        }

        .modal-header {
            border-radius: 20px 20px 0 0;
            padding: 1.5rem 2rem;
            border-bottom: none;
        }

        /* Alert */
        .alert-custom {
            border-radius: 16px;
            padding: 1.25rem 1.5rem;
            border: none;
            box-shadow: 0 4px 15px rgba(0, 0, 0, 0.1);
            animation: slideDown 0.3s ease;
        }

        @keyframes slideDown {
            from {
                opacity: 0;
                transform: translateY(-20px);
            }
            to {
                opacity: 1;
                transform: translateY(0);
            }
        }

        /* Filter Section */
        .filter-section {
            background: #f8fafc;
            padding: 1.5rem;
            border-radius: 16px;
            margin-bottom: 2rem;
        }

        /* Responsive */
        @media (max-width: 768px) {
            .sidebar {
                transform: translateX(-100%);
            }

            .sidebar.active {
                transform: translateX(0);
            }

            .main-content {
                margin-left: 0;
            }

            .mobile-toggle {
                display: block !important;
            }
        }

        .mobile-toggle {
            display: none;
            position: fixed;
            bottom: 2rem;
            right: 2rem;
            width: 60px;
            height: 60px;
            background: linear-gradient(135deg, var(--primary-color), var(--secondary-color));
            border-radius: 50%;
            box-shadow: 0 8px 20px rgba(99, 102, 241, 0.4);
            z-index: 999;
            border: none;
            color: white;
            font-size: 1.5rem;
        }

        /* Scrollbar */
        ::-webkit-scrollbar {
            width: 10px;
        }

        ::-webkit-scrollbar-track {
            background: rgba(255, 255, 255, 0.1);
        }

        ::-webkit-scrollbar-thumb {
            background: linear-gradient(135deg, var(--primary-color), var(--secondary-color));
            border-radius: 10px;
        }

        /* Empty State */
        .empty-state {
            text-align: center;
            padding: 4rem 2rem;
        }

        .empty-state-icon {
            width: 120px;
            height: 120px;
            background: linear-gradient(135deg, var(--primary-color), var(--secondary-color));
            border-radius: 50%;
            display: flex;
            align-items: center;
            justify-content: center;
            margin: 0 auto 2rem;
        }

        .empty-state-icon i {
            font-size: 3rem;
            color: white;
        }
    </style>
</head>
<body>

    <!-- Sidebar -->
    <div class="sidebar" id="sidebar">
        <div class="logo-container">
            <div class="logo-placeholder">
                <i class="fas fa-building"></i>
            </div>
            <h5 class="text-white fw-bold mb-1">KOPERASI</h5>
            <p class="text-white-50 mb-0" style="font-size: 0.875rem;">PUSDATIN</p>
        </div>

        <nav class="sidebar-menu">
            <div class="menu-section-title">Menu Utama</div>
            
            <a href="?page=dashboard" class="menu-item <?= $current_page == 'dashboard' ? 'active' : '' ?>">
                <i class="fas fa-home"></i>
                <span>Dashboard</span>
            </a>

            <div class="menu-section-title">Manajemen</div>
            
            <a href="?page=shu_configuration" class="menu-item <?= $current_page == 'shu_configuration' ? 'active' : '' ?>">
                <i class="fas fa-cog"></i>
                <span>Konfigurasi SHU</span>
            </a>

            <a href="?page=approval_pinjaman" class="menu-item <?= $current_page == 'approval_pinjaman' ? 'active' : '' ?>">
                <i class="fas fa-file-invoice-dollar"></i>
                <span>Approval Pinjaman</span>
                <?php if ($stats_waiting['total'] > 0): ?>
                <span class="badge bg-warning text-dark ms-auto"><?= $stats_waiting['total'] ?></span>
                <?php endif; ?>
            </a>

            <a href="?page=potongan_bon" class="menu-item <?= $current_page == 'potongan_bon' ? 'active' : '' ?>">
                <i class="fas fa-receipt"></i>
                <span>Potongan Bon Defmart</span>
            </a>

            <a href="?page=potongan_pinjaman" class="menu-item <?= $current_page == 'potongan_pinjaman' ? 'active' : '' ?>">
                <i class="fas fa-money-bill-transfer"></i>
                <span>Potongan Pinjaman</span>
            </a>

            <div class="menu-section-title">Laporan</div>
            
            <a href="?page=laporan" class="menu-item <?= $current_page == 'laporan' ? 'active' : '' ?>">
                <i class="fas fa-chart-bar"></i>
                <span>Laporan Keuangan</span>
            </a>

            <div class="menu-section-title">Lainnya</div>
            
            <a href="logout.php" class="menu-item text-danger">
                <i class="fas fa-sign-out-alt"></i>
                <span>Logout</span>
            </a>
        </nav>
    </div>

    <!-- Mobile Toggle Button -->
    <button class="mobile-toggle" onclick="toggleSidebar()">
        <i class="fas fa-bars"></i>
    </button>

    <!-- Main Content -->
    <div class="main-content">
        
        <!-- Top Bar -->
        <div class="top-bar">
            <div>
                <h4 class="mb-1 fw-bold text-dark">
                    <?php
                    $page_titles = [
                        'dashboard' => 'Dashboard Overview',
                        'shu_configuration' => 'Konfigurasi SHU',
                        'approval_pinjaman' => 'Approval Pinjaman',
                        'potongan_bon' => 'Potongan Bon Defmart',
                        'potongan_pinjaman' => 'Potongan Pinjaman Anggota',
                        'laporan' => 'Laporan Keuangan'
                    ];
                    echo $page_titles[$current_page] ?? 'Dashboard';
                    ?>
                </h4>
                <p class="mb-0 text-muted">
                    <i class="far fa-calendar-alt me-2"></i>
                    <?= strftime('%A, %d %B %Y', time()) ?>
                </p>
            </div>
            <div class="d-flex align-items-center gap-3">
                <div class="text-end">
                    <p class="mb-0 text-muted small">Selamat Datang,</p>
                    <h6 class="mb-0 fw-bold text-dark"><?= htmlspecialchars($username) ?></h6>
                </div>
                <div class="position-relative">
                    <div style="width: 50px; height: 50px; background: linear-gradient(135deg, #6366f1, #8b5cf6); border-radius: 50%; display: flex; align-items: center; justify-content: center;">
                        <i class="fas fa-user text-white"></i>
                    </div>
                    <span class="position-absolute bottom-0 end-0 translate-middle p-1 bg-success border border-white rounded-circle"></span>
                </div>
            </div>
        </div>

        <!-- Alert Messages -->
        <?php if ($sukses): ?>
        <div class="alert alert-success alert-custom d-flex align-items-center mb-4" role="alert">
            <i class="fas fa-check-circle me-3 fs-4"></i>
            <div><?= $sukses ?></div>
        </div>
        <?php endif; ?>

        <?php if ($error): ?>
        <div class="alert alert-danger alert-custom d-flex align-items-center mb-4" role="alert">
            <i class="fas fa-exclamation-circle me-3 fs-4"></i>
            <div><?= $error ?></div>
        </div>
        <?php endif; ?>

        <!-- Content Based on Page -->
        <?php if ($current_page == 'dashboard'): ?>
            <!-- Dashboard Content -->
            <div class="row g-4 mb-4">
                <div class="col-md-3">
                    <div class="stat-card">
                        <div class="stat-card-icon" style="background: linear-gradient(135deg, #f59e0b, #d97706);">
                            <i class="fas fa-clock"></i>
                        </div>
                        <h6 class="text-muted mb-2">Menunggu Approval</h6>
                        <h2 class="fw-bold mb-0 text-warning"><?= number_format($stats_waiting['total']) ?></h2>
                        <p class="text-muted mb-0 mt-2" style="font-size: 0.875rem;">
                            <i class="fas fa-arrow-up text-warning me-1"></i>Pinjaman
                        </p>
                    </div>
                </div>
                <div class="col-md-3">
                    <div class="stat-card">
                        <div class="stat-card-icon" style="background: linear-gradient(135deg, #10b981, #059669);">
                            <i class="fas fa-check-circle"></i>
                        </div>
                        <h6 class="text-muted mb-2">Disetujui</h6>
                        <h2 class="fw-bold mb-0 text-success"><?= number_format($stats_approved['total']) ?></h2>
                        <p class="text-muted mb-0 mt-2" style="font-size: 0.875rem;">
                            <i class="fas fa-check text-success me-1"></i>Pinjaman
                        </p>
                    </div>
                </div>
                <div class="col-md-3">
                    <div class="stat-card">
                        <div class="stat-card-icon" style="background: linear-gradient(135deg, #ef4444, #dc2626);">
                            <i class="fas fa-times-circle"></i>
                        </div>
                        <h6 class="text-muted mb-2">Ditolak</h6>
                        <h2 class="fw-bold mb-0 text-danger"><?= number_format($stats_rejected['total']) ?></h2>
                        <p class="text-muted mb-0 mt-2" style="font-size: 0.875rem;">
                            <i class="fas fa-times text-danger me-1"></i>Pinjaman
                        </p>
                    </div>
                </div>
                <div class="col-md-3">
                    <div class="stat-card">
                        <div class="stat-card-icon" style="background: linear-gradient(135deg, #6366f1, #8b5cf6);">
                            <i class="fas fa-money-bill-wave"></i>
                        </div>
                        <h6 class="text-muted mb-2">Total Disetujui</h6>
                        <h5 class="fw-bold mb-0 text-primary">Rp <?= number_format($stats_nilai['total'] ?? 0, 0, ',', '.') ?></h5>
                        <p class="text-muted mb-0 mt-2" style="font-size: 0.875rem;">
                            <i class="fas fa-wallet text-primary me-1"></i>Nilai Pinjaman
                        </p>
                    </div>
                </div>
            </div>

            <!-- Quick Stats Row 2 -->
            <div class="row g-4 mb-4">
                <div class="col-md-6">
                    <div class="stat-card">
                        <div class="d-flex justify-content-between align-items-center mb-3">
                            <h6 class="fw-bold mb-0">Potongan Bon Defmart</h6>
                            <div class="stat-card-icon" style="background: linear-gradient(135deg, #3b82f6, #2563eb); width: 48px; height: 48px;">
                                <i class="fas fa-receipt" style="font-size: 1.25rem;"></i>
                            </div>
                        </div>
                        <div class="row">
                            <div class="col-6">
                                <p class="text-muted mb-1" style="font-size: 0.875rem;">Total Anggota</p>
                                <h4 class="fw-bold text-info"><?= number_format($stats_bon['total_anggota'] ?? 0) ?></h4>
                            </div>
                            <div class="col-6">
                                <p class="text-muted mb-1" style="font-size: 0.875rem;">Total Bon</p>
                                <h6 class="fw-bold text-info">Rp <?= number_format($stats_bon['total_bon'] ?? 0, 0, ',', '.') ?></h6>
                            </div>
                        </div>
                    </div>
                </div>
                <div class="col-md-6">
                    <div class="stat-card">
                        <div class="d-flex justify-content-between align-items-center mb-3">
                            <h6 class="fw-bold mb-0">Pinjaman Aktif</h6>
                            <div class="stat-card-icon" style="background: linear-gradient(135deg, #8b5cf6, #7c3aed); width: 48px; height: 48px;">
                                <i class="fas fa-hand-holding-usd" style="font-size: 1.25rem;"></i>
                            </div>
                        </div>
                        <div class="row">
                            <div class="col-12">
                                <p class="text-muted mb-1" style="font-size: 0.875rem;">Anggota dengan Pinjaman Aktif</p>
                                <h4 class="fw-bold" style="background: linear-gradient(135deg, #8b5cf6, #7c3aed); -webkit-background-clip: text; -webkit-text-fill-color: transparent;"><?= number_format($stats_pinjaman_aktif['total'] ?? 0) ?></h4>
                            </div>
                        </div>
                    </div>
                </div>
            </div>

            <!-- Quick Actions -->
            <div class="content-card">
                <h5 class="fw-bold mb-4">
                    <i class="fas fa-bolt me-2" style="color: #f59e0b;"></i>Aksi Cepat
                </h5>
                <div class="row g-3">
                    <div class="col-md-3">
                        <a href="?page=approval_pinjaman&status=waiting" class="btn btn-custom btn-primary-custom w-100 d-flex align-items-center justify-content-between">
                            <span><i class="fas fa-tasks me-2"></i>Approval Pinjaman</span>
                            <?php if ($stats_waiting['total'] > 0): ?>
                            <span class="badge bg-white text-primary"><?= $stats_waiting['total'] ?></span>
                            <?php endif; ?>
                        </a>
                    </div>
                    <div class="col-md-3">
                        <a href="?page=potongan_bon" class="btn btn-custom btn-info-custom w-100">
                            <i class="fas fa-receipt me-2"></i>Kelola Bon
                        </a>
                    </div>
                    <div class="col-md-3">
                        <a href="?page=potongan_pinjaman" class="btn btn-custom btn-success-custom w-100">
                            <i class="fas fa-money-bill-transfer me-2"></i>Potongan Gaji
                        </a>
                    </div>
                    <div class="col-md-3">
                        <a href="?page=shu_configuration" class="btn btn-custom btn-danger-custom w-100">
                            <i class="fas fa-cog me-2"></i>Konfigurasi SHU
                        </a>
                    </div>
                </div>
            </div>

        <?php elseif ($current_page == 'shu_configuration'): ?>
            <!-- SHU Configuration Content -->
            <div class="content-card">
                <div class="d-flex justify-content-between align-items-center mb-4">
                    <h5 class="fw-bold mb-0">
                        <i class="fas fa-cog me-2 text-primary"></i>Konfigurasi Sisa Hasil Usaha (SHU)
                    </h5>
                    <button class="btn btn-custom btn-primary-custom">
                        <i class="fas fa-plus me-2"></i>Tambah Konfigurasi
                    </button>
                </div>
                
                <div class="empty-state">
                    <div class="empty-state-icon">
                        <i class="fas fa-cog"></i>
                    </div>
                    <h4 class="fw-bold mb-3">Konfigurasi SHU</h4>
                    <p class="text-muted mb-4">Halaman ini akan menampilkan pengaturan untuk perhitungan dan distribusi Sisa Hasil Usaha (SHU) kepada anggota koperasi.</p>
                    <button class="btn btn-custom btn-primary-custom">
                        <i class="fas fa-rocket me-2"></i>Mulai Konfigurasi
                    </button>
                </div>
            </div>

        <?php elseif ($current_page == 'approval_pinjaman'): ?>
            <!-- Approval Pinjaman Content -->
            <div class="filter-section">
                <form method="GET" class="row g-3 align-items-end">
                    <input type="hidden" name="page" value="approval_pinjaman">
                    <div class="col-md-3">
                        <label class="form-label fw-semibold">
                            <i class="fas fa-filter me-2 text-primary"></i>Status
                        </label>
                        <select name="status" class="form-select" onchange="this.form.submit()">
                            <option value="waiting" <?= $filter_status == 'waiting' ? 'selected' : '' ?>>Menunggu Approval</option>
                            <option value="approved" <?= $filter_status == 'approved' ? 'selected' : '' ?>>Disetujui</option>
                            <option value="rejected" <?= $filter_status == 'rejected' ? 'selected' : '' ?>>Ditolak</option>
                        </select>
                    </div>
                    <div class="col-md-6">
                        <label class="form-label fw-semibold">
                            <i class="fas fa-search me-2 text-primary"></i>Cari Anggota
                        </label>
                        <input type="text" name="search" value="<?= htmlspecialchars($filter_search) ?>" class="form-control" placeholder="Cari berdasarkan nama atau NIP...">
                    </div>
                    <div class="col-md-3">
                        <button type="submit" class="btn btn-custom btn-primary-custom w-100">
                            <i class="fas fa-search me-2"></i>Cari
                        </button>
                    </div>
                </form>
            </div>

            <?php if (count($data_pinjaman) > 0): ?>
            <div class="row g-4">
                <?php foreach ($data_pinjaman as $p): 
                    $is_waiting = !$p['approved_by_payroll'] && $p['status'] != 'ditolak';
                    $is_approved = $p['approved_by_payroll'] && $p['status'] == 'disetujui';
                    $is_rejected = $p['status'] == 'ditolak';
                    
                    $header_class = $is_waiting ? 'warning' : ($is_approved ? 'success' : 'danger');
                    $header_text = $is_waiting ? 'Menunggu Approval Payroll' : ($is_approved ? 'Disetujui' : 'Ditolak');
                    $header_icon = $is_waiting ? 'fa-hourglass-half' : ($is_approved ? 'fa-check-double' : 'fa-times-circle');
                ?>
                
                <div class="col-lg-6">
                    <div class="loan-card">
                        <div class="bg-<?= $header_class ?> text-white p-3">
                            <div class="d-flex justify-content-between align-items-center">
                                <span class="fw-bold">
                                    <i class="fas <?= $header_icon ?> me-2"></i><?= $header_text ?>
                                </span>
                                <span class="badge bg-white bg-opacity-25">
                                    #PJM-<?= str_pad($p['id_pinjaman'], 3, '0', STR_PAD_LEFT) ?>
                                </span>
                            </div>
                        </div>
                        
                        <div class="p-4">
                            <div class="d-flex align-items-center mb-3">
                                <div style="width: 50px; height: 50px; background: linear-gradient(135deg, #6366f1, #8b5cf6); border-radius: 12px; display: flex; align-items: center; justify-content: center; margin-right: 1rem;">
                                    <i class="fas fa-user text-white"></i>
                                </div>
                                <div>
                                    <h6 class="fw-bold mb-0"><?= htmlspecialchars($p['nama']) ?></h6>
                                    <small class="text-muted">NIP: <?= htmlspecialchars($p['nip']) ?></small>
                                </div>
                            </div>

                            <div class="row g-3 mb-3">
                                <div class="col-6">
                                    <div class="bg-light p-3 rounded-3">
                                        <small class="text-muted d-block mb-1">Jumlah Pinjaman</small>
                                        <h6 class="fw-bold mb-0 text-primary">Rp <?= number_format($p['jumlah_pinjaman'], 0, ',', '.') ?></h6>
                                    </div>
                                </div>
                                <div class="col-6">
                                    <div class="bg-light p-3 rounded-3">
                                        <small class="text-muted d-block mb-1">Lama Bayar</small>
                                        <h6 class="fw-bold mb-0 text-info"><?= $p['lama_bayar'] ?> Bulan</h6>
                                    </div>
                                </div>
                                <div class="col-6">
                                    <div class="bg-light p-3 rounded-3">
                                        <small class="text-muted d-block mb-1">Angsuran/Bulan</small>
                                        <h6 class="fw-bold mb-0 text-success">Rp <?= number_format($p['angsuran_perbulan'], 0, ',', '.') ?></h6>
                                    </div>
                                </div>
                                <div class="col-6">
                                    <div class="bg-light p-3 rounded-3">
                                        <small class="text-muted d-block mb-1">Total Bayar</small>
                                        <h6 class="fw-bold mb-0 text-warning">Rp <?= number_format($p['total_bayar'], 0, ',', '.') ?></h6>
                                    </div>
                                </div>
                            </div>

                            <?php if ($p['dokumen_pendukung']): ?>
                            <div class="alert alert-info mb-3">
                                <i class="fas fa-file-pdf me-2"></i>
                                <a href="uploads/dokumen_pinjaman/<?= $p['dokumen_pendukung'] ?>" target="_blank" class="text-info fw-semibold">
                                    Lihat Dokumen <i class="fas fa-external-link-alt ms-1"></i>
                                </a>
                            </div>
                            <?php endif; ?>

                            <div class="alert alert-success mb-3">
                                <small class="fw-semibold d-block mb-1">
                                    <i class="fas fa-check-circle me-1"></i>Disetujui oleh Admin
                                </small>
                                <small class="text-success"><?= htmlspecialchars($p['admin_name']) ?> • <?= date('d M Y, H:i', strtotime($p['tanggal_approve_admin'])) ?></small>
                                <?php if ($p['catatan_admin']): ?>
                                <small class="d-block mt-1 fst-italic">"<?= htmlspecialchars($p['catatan_admin']) ?>"</small>
                                <?php endif; ?>
                            </div>

                            <?php if ($p['approved_by_payroll']): ?>
                            <div class="alert alert-<?= $is_approved ? 'success' : 'danger' ?> mb-3">
                                <small class="fw-semibold d-block mb-1">
                                    <i class="fas fa-check-circle me-1"></i><?= $is_approved ? 'Disetujui' : 'Ditolak' ?> oleh Payroll
                                </small>
                                <small><?= htmlspecialchars($p['payroll_name']) ?> • <?= date('d M Y, H:i', strtotime($p['tanggal_approve_payroll'])) ?></small>
                                <?php if ($p['catatan_payroll']): ?>
                                <small class="d-block mt-1 fst-italic">"<?= htmlspecialchars($p['catatan_payroll']) ?>"</small>
                                <?php endif; ?>
                            </div>
                            <?php endif; ?>

                            <div class="d-flex gap-2">
                                <?php if ($is_waiting): ?>
                                <button onclick="openApproveModal(<?= $p['id_pinjaman'] ?>, '<?= htmlspecialchars($p['nama']) ?>')" class="btn btn-custom btn-success-custom flex-fill">
                                    <i class="fas fa-check me-2"></i>Setujui
                                </button>
                                <button onclick="openRejectModal(<?= $p['id_pinjaman'] ?>, '<?= htmlspecialchars($p['nama']) ?>')" class="btn btn-custom btn-danger-custom flex-fill">
                                    <i class="fas fa-times me-2"></i>Tolak
                                </button>
                                <?php endif; ?>
                                
                                <button onclick="openDetailModal(<?= $p['id_pinjaman'] ?>)" class="btn btn-custom btn-info-custom flex-fill">
                                    <i class="fas fa-eye me-2"></i>Detail
                                </button>
                            </div>
                        </div>
                    </div>
                </div>
                
                <?php endforeach; ?>
            </div>
            <?php else: ?>
            <div class="content-card">
                <div class="empty-state">
                    <div class="empty-state-icon">
                        <i class="fas fa-inbox"></i>
                    </div>
                    <h4 class="fw-bold mb-3">Tidak Ada Data</h4>
                    <p class="text-muted">Belum ada pinjaman yang memerlukan approval pada filter ini.</p>
                </div>
            </div>
            <?php endif; ?>

        <?php elseif ($current_page == 'potongan_bon'): ?>
            <!-- Potongan Bon Defmart Content -->
            <div class="content-card">
                <div class="d-flex justify-content-between align-items-center mb-4">
                    <h5 class="fw-bold mb-0">
                        <i class="fas fa-receipt me-2 text-primary"></i>Kelola Potongan Bon Defmart
                    </h5>
                    <div class="d-flex gap-2">
                        <button class="btn btn-custom btn-success-custom">
                            <i class="fas fa-file-excel me-2"></i>Export Excel
                        </button>
                        <button class="btn btn-custom btn-primary-custom">
                            <i class="fas fa-calculator me-2"></i>Proses Potongan
                        </button>
                    </div>
                </div>
                
                <div class="filter-section mb-4">
                    <form class="row g-3 align-items-end">
                        <input type="hidden" name="page" value="potongan_bon">
                        <div class="col-md-3">
                            <label class="form-label fw-semibold">
                                <i class="fas fa-filter me-2 text-primary"></i>Status Pembayaran
                            </label>
                            <select name="status" class="form-select">
                                <option value="">Semua Status</option>
                                <option value="belum_lunas">Belum Lunas</option>
                                <option value="lunas">Lunas</option>
                            </select>
                        </div>
                        <div class="col-md-3">
                            <label class="form-label fw-semibold">
                                <i class="fas fa-calendar me-2 text-primary"></i>Bulan
                            </label>
                            <input type="month" name="bulan" class="form-control">
                        </div>
                        <div class="col-md-4">
                            <label class="form-label fw-semibold">
                                <i class="fas fa-search me-2 text-primary"></i>Cari Anggota
                            </label>
                            <input type="text" name="search" class="form-control" placeholder="Nama atau NIP...">
                        </div>
                        <div class="col-md-2">
                            <button type="submit" class="btn btn-custom btn-primary-custom w-100">
                                <i class="fas fa-search me-2"></i>Filter
                            </button>
                        </div>
                    </form>
                </div>

                <div class="empty-state">
                    <div class="empty-state-icon">
                        <i class="fas fa-receipt"></i>
                    </div>
                    <h4 class="fw-bold mb-3">Kelola Potongan Bon</h4>
                    <p class="text-muted mb-4">Halaman ini menampilkan daftar anggota yang memiliki transaksi bon di Defmart dan memproses potongan otomatis dari gaji bulanan.</p>
                </div>
            </div>

        <?php elseif ($current_page == 'potongan_pinjaman'): ?>
            <!-- Potongan Pinjaman Content -->
            <div class="content-card">
                <div class="d-flex justify-content-between align-items-center mb-4">
                    <h5 class="fw-bold mb-0">
                        <i class="fas fa-money-bill-transfer me-2 text-primary"></i>Kelola Potongan Pinjaman
                    </h5>
                    <div class="d-flex gap-2">
                        <button class="btn btn-custom btn-success-custom">
                            <i class="fas fa-file-excel me-2"></i>Export Excel
                        </button>
                        <button class="btn btn-custom btn-primary-custom">
                            <i class="fas fa-calculator me-2"></i>Proses Potongan
                        </button>
                    </div>
                </div>
                
                <div class="filter-section mb-4">
                    <form class="row g-3 align-items-end">
                        <input type="hidden" name="page" value="potongan_pinjaman">
                        <div class="col-md-3">
                            <label class="form-label fw-semibold">
                                <i class="fas fa-filter me-2 text-primary"></i>Status Pinjaman
                            </label>
                            <select name="status" class="form-select">
                                <option value="">Semua Status</option>
                                <option value="aktif">Aktif</option>
                                <option value="lunas">Lunas</option>
                            </select>
                        </div>
                        <div class="col-md-3">
                            <label class="form-label fw-semibold">
                                <i class="fas fa-calendar me-2 text-primary"></i>Periode
                            </label>
                            <input type="month" name="periode" class="form-control" value="<?= date('Y-m') ?>">
                        </div>
                        <div class="col-md-4">
                            <label class="form-label fw-semibold">
                                <i class="fas fa-search me-2 text-primary"></i>Cari Anggota
                            </label>
                            <input type="text" name="search" class="form-control" placeholder="Nama atau NIP...">
                        </div>
                        <div class="col-md-2">
                            <button type="submit" class="btn btn-custom btn-primary-custom w-100">
                                <i class="fas fa-search me-2"></i>Filter
                            </button>
                        </div>
                    </form>
                </div>

                <div class="empty-state">
                    <div class="empty-state-icon">
                        <i class="fas fa-money-bill-transfer"></i>
                    </div>
                    <h4 class="fw-bold mb-3">Kelola Potongan Pinjaman</h4>
                    <p class="text-muted mb-4">Halaman ini menampilkan daftar anggota yang memiliki pinjaman aktif dan memproses potongan angsuran otomatis dari gaji bulanan.</p>
                </div>
            </div>

        <?php elseif ($current_page == 'laporan'): ?>
            <!-- Laporan Content -->
            <div class="content-card">
                <div class="d-flex justify-content-between align-items-center mb-4">
                    <h5 class="fw-bold mb-0">
                        <i class="fas fa-chart-bar me-2 text-primary"></i>Laporan Keuangan
                    </h5>
                    <button class="btn btn-custom btn-success-custom">
                        <i class="fas fa-download me-2"></i>Download Laporan
                    </button>
                </div>
                
                <div class="empty-state">
                    <div class="empty-state-icon">
                        <i class="fas fa-chart-bar"></i>
                    </div>
                    <h4 class="fw-bold mb-3">Laporan Keuangan</h4>
                    <p class="text-muted mb-4">Halaman ini menampilkan berbagai laporan keuangan termasuk laporan pinjaman, potongan, dan SHU.</p>
                </div>
            </div>

        <?php endif; ?>

    </div>

    <!-- Modal Approve -->
    <div class="modal fade" id="modalApprove" tabindex="-1">
        <div class="modal-dialog modal-dialog-centered">
            <div class="modal-content">
                <div class="modal-header" style="background: linear-gradient(135deg, #10b981, #059669);">
                    <h5 class="modal-title text-white">
                        <i class="fas fa-check-circle me-2"></i>Setujui Pinjaman
                    </h5>
                    <button type="button" class="btn-close btn-close-white" data-bs-dismiss="modal"></button>
                </div>
                <form method="POST">
                    <div class="modal-body p-4">
                        <input type="hidden" name="id_pinjaman" id="approve_id_pinjaman">
                        
                        <div class="alert alert-success mb-4">
                            <h6 class="fw-bold mb-1">Anggota:</h6>
                            <p id="approve_nama_anggota" class="mb-0 fw-semibold"></p>
                        </div>

                        <div class="mb-3">
                            <label class="form-label fw-semibold">
                                <i class="fas fa-comment me-2 text-primary"></i>Catatan (Opsional)
                            </label>
                            <textarea name="catatan_payroll" rows="3" class="form-control" placeholder="Tambahkan catatan jika diperlukan..."></textarea>
                        </div>
                    </div>
                    <div class="modal-footer">
                        <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Batal</button>
                        <button type="submit" name="approve_payroll" class="btn btn-custom btn-success-custom">
                            <i class="fas fa-check me-2"></i>Setujui Pinjaman
                        </button>
                    </div>
                </form>
            </div>
        </div>
    </div>

    <!-- Modal Reject -->
    <div class="modal fade" id="modalReject" tabindex="-1">
        <div class="modal-dialog modal-dialog-centered">
            <div class="modal-content">
                <div class="modal-header bg-danger">
                    <h5 class="modal-title text-white">
                        <i class="fas fa-times-circle me-2"></i>Tolak Pinjaman
                    </h5>
                    <button type="button" class="btn-close btn-close-white" data-bs-dismiss="modal"></button>
                </div>
                <form method="POST">
                    <div class="modal-body p-4">
                        <input type="hidden" name="id_pinjaman" id="reject_id_pinjaman">
                        
                        <div class="alert alert-danger mb-4">
                            <h6 class="fw-bold mb-1">Anggota:</h6>
                            <p id="reject_nama_anggota" class="mb-0 fw-semibold"></p>
                        </div>

                        <div class="mb-3">
                            <label class="form-label fw-semibold">
                                <i class="fas fa-comment me-2 text-danger"></i>Alasan Penolakan <span class="text-danger">*</span>
                            </label>
                            <textarea name="catatan_payroll" rows="3" required class="form-control" placeholder="Jelaskan alasan penolakan..."></textarea>
                        </div>
                    </div>
                    <div class="modal-footer">
                        <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Batal</button>
                        <button type="submit" name="reject_payroll" class="btn btn-custom btn-danger-custom">
                            <i class="fas fa-times me-2"></i>Tolak Pinjaman
                        </button>
                    </div>
                </form>
            </div>
        </div>
    </div>

    <!-- Modal Detail -->
    <div class="modal fade" id="modalDetail" tabindex="-1">
        <div class="modal-dialog modal-dialog-centered modal-lg modal-dialog-scrollable">
            <div class="modal-content">
                <div class="modal-header" style="background: linear-gradient(135deg, #6366f1, #8b5cf6);">
                    <h5 class="modal-title text-white">
                        <i class="fas fa-info-circle me-2"></i>Detail Pinjaman
                    </h5>
                    <button type="button" class="btn-close btn-close-white" data-bs-dismiss="modal"></button>
                </div>
                <div class="modal-body p-4" id="detail_content">
                    <!-- Content will be filled via JavaScript -->
                </div>
            </div>
        </div>
    </div>

    <!-- Bootstrap JS -->
    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/js/bootstrap.bundle.min.js"></script>

    <script>
        const dataPinjaman = <?= json_encode($data_pinjaman) ?>;

        function toggleSidebar() {
            document.getElementById('sidebar').classList.toggle('active');
        }

        function openApproveModal(id, nama) {
            document.getElementById('approve_id_pinjaman').value = id;
            document.getElementById('approve_nama_anggota').textContent = nama;
            new bootstrap.Modal(document.getElementById('modalApprove')).show();
        }

        function openRejectModal(id, nama) {
            document.getElementById('reject_id_pinjaman').value = id;
            document.getElementById('reject_nama_anggota').textContent = nama;
            new bootstrap.Modal(document.getElementById('modalReject')).show();
        }

        function openDetailModal(id) {
            const pinjaman = dataPinjaman.find(p => p.id_pinjaman == id);
            
            if (!pinjaman) return;
            
            let html = `
                <div class="mb-4 p-4 rounded-3" style="background: linear-gradient(135deg, #6366f1, #8b5cf6);">
                    <div class="d-flex justify-content-between align-items-center text-white">
                        <div>
                            <h4 class="fw-bold mb-1">${pinjaman.nama}</h4>
                            <p class="mb-0 opacity-75">NIP: ${pinjaman.nip}</p>
                        </div>
                        <div class="text-end">
                            <small class="d-block opacity-75">No. Pinjaman</small>
                            <h5 class="fw-bold mb-0">#PJM-${String(pinjaman.id_pinjaman).padStart(3, '0')}</h5>
                        </div>
                    </div>
                </div>

                <div class="row g-3 mb-4">
                    <div class="col-6">
                        <div class="bg-light p-3 rounded-3">
                            <small class="text-muted d-block mb-1">Jumlah Pinjaman</small>
                            <h6 class="fw-bold mb-0 text-primary">Rp ${parseFloat(pinjaman.jumlah_pinjaman).toLocaleString('id-ID')}</h6>
                        </div>
                    </div>
                    <div class="col-6">
                        <div class="bg-light p-3 rounded-3">
                            <small class="text-muted d-block mb-1">Bunga</small>
                            <h6 class="fw-bold mb-0 text-info">${pinjaman.bunga_persen}%</h6>
                        </div>
                    </div>
                    <div class="col-6">
                        <div class="bg-light p-3 rounded-3">
                            <small class="text-muted d-block mb-1">Total Bayar</small>
                            <h6 class="fw-bold mb-0 text-success">Rp ${parseFloat(pinjaman.total_bayar).toLocaleString('id-ID')}</h6>
                        </div>
                    </div>
                    <div class="col-6">
                        <div class="bg-light p-3 rounded-3">
                            <small class="text-muted d-block mb-1">Lama Bayar</small>
                            <h6 class="fw-bold mb-0 text-warning">${pinjaman.lama_bayar} Bulan</h6>
                        </div>
                    </div>
                    <div class="col-12">
                        <div class="bg-light p-3 rounded-3">
                            <small class="text-muted d-block mb-1">Angsuran Per Bulan</small>
                            <h5 class="fw-bold mb-0 text-danger">Rp ${parseFloat(pinjaman.angsuran_perbulan).toLocaleString('id-ID')}</h5>
                        </div>
                    </div>
                </div>

                ${pinjaman.dokumen_pendukung ? `
                <div class="alert alert-info mb-4">
                    <h6 class="fw-bold mb-2">Dokumen Pendukung</h6>
                    <a href="uploads/dokumen_pinjaman/${pinjaman.dokumen_pendukung}" target="_blank" class="btn btn-sm btn-info">
                        <i class="fas fa-file-pdf me-2"></i>Lihat Dokumen
                        <i class="fas fa-external-link-alt ms-2"></i>
                    </a>
                </div>
                ` : ''}

                <div class="timeline">
                    <div class="alert alert-success mb-3">
                        <div class="d-flex align-items-start">
                            <i class="fas fa-check-circle text-success fs-4 me-3 mt-1"></i>
                            <div>
                                <h6 class="fw-bold mb-1">Disetujui oleh Admin</h6>
                                <p class="mb-1">${pinjaman.admin_name}</p>
                                <small class="text-muted">${new Date(pinjaman.tanggal_approve_admin).toLocaleString('id-ID')}</small>
                                ${pinjaman.catatan_admin ? `<p class="mt-2 mb-0 fst-italic">"${pinjaman.catatan_admin}"</p>` : ''}
                            </div>
                        </div>
                    </div>

                    ${pinjaman.approved_by_payroll ? `
                    <div class="alert alert-${pinjaman.status == 'disetujui' ? 'success' : 'danger'}">
                        <div class="d-flex align-items-start">
                            <i class="fas fa-check-circle text-${pinjaman.status == 'disetujui' ? 'success' : 'danger'} fs-4 me-3 mt-1"></i>
                            <div>
                                <h6 class="fw-bold mb-1">${pinjaman.status == 'disetujui' ? 'Disetujui' : 'Ditolak'} oleh Payroll</h6>
                                <p class="mb-1">${pinjaman.payroll_name}</p>
                                <small class="text-muted">${new Date(pinjaman.tanggal_approve_payroll).toLocaleString('id-ID')}</small>
                                ${pinjaman.catatan_payroll ? `<p class="mt-2 mb-0 fst-italic">"${pinjaman.catatan_payroll}"</p>` : ''}
                            </div>
                        </div>
                    </div>
                    ` : ''}
                </div>
            `;
            
            document.getElementById('detail_content').innerHTML = html;
            new bootstrap.Modal(document.getElementById('modalDetail')).show();
        }

        // Auto dismiss alerts
        setTimeout(() => {
            const alerts = document.querySelectorAll('.alert-custom');
            alerts.forEach(alert => {
                alert.style.transition = 'opacity 0.5s';
                alert.style.opacity = '0';
                setTimeout(() => alert.remove(), 500);
            });
        }, 5000);
    </script>

</body>
</html>