<?php
session_start();
include 'config.php';

header('Content-Type: application/json');

// Check if user is authorized
if (!isset($_SESSION['role']) || $_SESSION['id_role'] != 7) {
    echo json_encode(['error' => 'Unauthorized']);
    exit;
}

// Get pinjaman ID
$id_pinjaman = isset($_GET['id']) ? intval($_GET['id']) : 0;

if ($id_pinjaman <= 0) {
    echo json_encode(['error' => 'Invalid ID']);
    exit;
}

// Query to get loan details with member information
$query = "SELECT 
    p.id_pinjaman,
    p.id_anggota,
    p.jumlah_pinjaman,
    p.bunga_persen,
    p.total_bayar,
    p.lama_bayar,
    p.angsuran_perbulan,
    p.status,
    p.tanggal_pengajuan,
    p.dokumen_pendukung,
    a.nama,
    a.nip,
    a.jabatan,
    a.email,
    a.no_telp,
    a.alamat
FROM pinjaman p
JOIN anggota a ON p.id_anggota = a.id_anggota
WHERE p.id_pinjaman = ? AND p.status = 'pending' AND p.jumlah_pinjaman <= 10000000";

$stmt = $conn->prepare($query);
if (!$stmt) {
    echo json_encode(['error' => 'Database error: ' . $conn->error]);
    exit;
}

$stmt->bind_param("i", $id_pinjaman);
$stmt->execute();
$result = $stmt->get_result();

if ($result->num_rows === 0) {
    echo json_encode(['error' => 'Data tidak ditemukan atau sudah diproses']);
    exit;
}

$data = $result->fetch_assoc();

// Calculate loan details menggunakan data dari database
// Jika total_bayar dan angsuran_perbulan sudah ada di database, gunakan itu
// Jika tidak, hitung ulang
if ($data['total_bayar'] && $data['angsuran_perbulan']) {
    $total_bayar = floatval($data['total_bayar']);
    $angsuran_per_bulan = floatval($data['angsuran_perbulan']);
    $bunga_total = $total_bayar - floatval($data['jumlah_pinjaman']);
} else {
    // Hitung jika belum ada di database
    $bunga_per_bulan = 1.5; // 1.5% per bulan
    $bunga_total = (floatval($data['jumlah_pinjaman']) * $bunga_per_bulan / 100) * intval($data['lama_bayar']);
    $total_bayar = floatval($data['jumlah_pinjaman']) + $bunga_total;
    $angsuran_per_bulan = $total_bayar / intval($data['lama_bayar']);
}

// Prepare document URL if exists
$dokumen_url = '';
$dokumen_exists = false;

if (!empty($data['dokumen_pendukung'])) {
    // Coba beberapa kemungkinan path dokumen
    $possible_paths = [
        'uploads/dokumen_pinjaman/' . $data['dokumen_pendukung'],
        'uploads/pinjaman/' . $data['dokumen_pendukung'],
        'uploads/' . $data['dokumen_pendukung'],
        'dokumen/' . $data['dokumen_pendukung'],
        'dokumen_pinjaman/' . $data['dokumen_pendukung'],
        $data['dokumen_pendukung'] // Path lengkap dari database
    ];
    
    foreach ($possible_paths as $path) {
        if (file_exists($path)) {
            $dokumen_url = $path;
            $dokumen_exists = true;
            break;
        }
    }
    
    // Jika file tidak ditemukan, tetap set URL (untuk debugging)
    if (!$dokumen_exists) {
        $dokumen_url = 'uploads/dokumen_pinjaman/' . $data['dokumen_pendukung'];
    }
}

// Prepare response
$response = [
    'id_pinjaman' => intval($data['id_pinjaman']),
    'id_anggota' => intval($data['id_anggota']),
    'nama' => $data['nama'] ?? '-',
    'nip' => $data['nip'] ?? '-',
    'jabatan' => $data['jabatan'] ?? '-',
    'email' => $data['email'] ?? '-',
    'no_telp' => $data['no_telp'] ?? '-',
    'alamat' => $data['alamat'] ?? '-',
    'jumlah_pinjaman' => floatval($data['jumlah_pinjaman']),
    'lama_bayar' => intval($data['lama_bayar']),
    'bunga_persen' => floatval($data['bunga_persen']),
    'bunga_total' => $bunga_total,
    'total_bayar' => $total_bayar,
    'angsuran_per_bulan' => $angsuran_per_bulan,
    'tanggal_pengajuan' => $data['tanggal_pengajuan'],
    'status' => $data['status'],
    'dokumen_pendukung' => $data['dokumen_pendukung'],
    'dokumen_url' => $dokumen_url,
    'dokumen_exists' => $dokumen_exists
];

echo json_encode($response);

$stmt->close();
$conn->close();
?>