<?php
session_start();
include 'config.php';

error_reporting(E_ALL);
ini_set('display_errors', 1);

// Cek apakah user login dan role = Kepala Unit (id_role = 7)
if (!isset($_SESSION['role']) || $_SESSION['id_role'] != 7) {
    header("Location: index.php");
    exit;
}

$id_unit = 1; 
$nama_unit = "Simpan Pinjam";

// Array nama bulan
$bulan_nama = ['', 'Januari', 'Februari', 'Maret', 'April', 'Mei', 'Juni', 
              'Juli', 'Agustus', 'September', 'Oktober', 'November', 'Desember'];

// Proses input simpanan sukarela
if (isset($_POST['simpan_sukarela'])) {
    $id_anggota = intval($_POST['id_anggota']);
    $nominal = floatval(str_replace(['.', ','], ['', '.'], $_POST['nominal']));
    $tanggal_setor = $_POST['tanggal_setor'];
    $keterangan = trim($_POST['keterangan']);
    
    // Validasi
    if ($id_anggota <= 0) {
        header("Location: input_simpanan_sukarela.php?error=anggota_kosong");
        exit;
    }
    
    if ($nominal <= 0) {
        header("Location: input_simpanan_sukarela.php?error=nominal_invalid");
        exit;
    }
    
    if (empty($tanggal_setor)) {
        header("Location: input_simpanan_sukarela.php?error=tanggal_kosong");
        exit;
    }
    
    // Ambil tahun dan bulan dari tanggal setor
    $tahun = intval(date('Y', strtotime($tanggal_setor)));
    $bulan = intval(date('n', strtotime($tanggal_setor)));
    
    $conn->begin_transaction();
    
    try {
        // 1. Insert ke tabel simpanan dengan jenis 'sukarela'
        $stmt_simpanan = $conn->prepare("INSERT INTO simpanan (id_anggota, jenis, nominal, tanggal_setor, keterangan) 
            VALUES (?, 'sukarela', ?, ?, ?)");
        
        if (!$stmt_simpanan) {
            throw new Exception("Prepare simpanan gagal: " . $conn->error);
        }
        
        if (empty($keterangan)) {
            $keterangan = "Simpanan Sukarela - " . $bulan_nama[$bulan] . " " . $tahun;
        }
        
        $stmt_simpanan->bind_param("idss", $id_anggota, $nominal, $tanggal_setor, $keterangan);
        
        if (!$stmt_simpanan->execute()) {
            throw new Exception("Error inserting simpanan: " . $stmt_simpanan->error);
        }
        
        $stmt_simpanan->close();
        
        // 2. Cek apakah sudah ada record tabungan untuk bulan dan tahun ini
        $stmt_cek_tabungan = $conn->prepare("SELECT id_tabungan, setoran, pengambilan, saldo FROM tabungan_usipa 
            WHERE id_anggota = ? AND tahun = ? AND bulan = ?");
        
        if (!$stmt_cek_tabungan) {
            throw new Exception("Prepare cek tabungan gagal: " . $conn->error);
        }
        
        $stmt_cek_tabungan->bind_param("iii", $id_anggota, $tahun, $bulan);
        $stmt_cek_tabungan->execute();
        $result_tabungan = $stmt_cek_tabungan->get_result();
        
        // 3. Ambil saldo bulan sebelumnya
        $stmt_saldo = $conn->prepare("SELECT saldo FROM tabungan_usipa 
            WHERE id_anggota = ? AND (tahun < ? OR (tahun = ? AND bulan < ?))
            ORDER BY tahun DESC, bulan DESC LIMIT 1");
        
        if (!$stmt_saldo) {
            throw new Exception("Prepare saldo gagal: " . $conn->error);
        }
        
        $stmt_saldo->bind_param("iiii", $id_anggota, $tahun, $tahun, $bulan);
        $stmt_saldo->execute();
        $result_saldo = $stmt_saldo->get_result();
        
        $saldo_sebelumnya = 0;
        if ($result_saldo && $result_saldo->num_rows > 0) {
            $row_saldo = $result_saldo->fetch_assoc();
            $saldo_sebelumnya = floatval($row_saldo['saldo']);
        }
        
        $stmt_saldo->close();
        
        // 4. Update atau Insert ke tabel tabungan_usipa
        if ($result_tabungan && $result_tabungan->num_rows > 0) {
            // Update existing record - tambahkan ke setoran yang sudah ada
            $data_tabungan = $result_tabungan->fetch_assoc();
            $setoran_baru = floatval($data_tabungan['setoran']) + $nominal;
            $pengambilan = floatval($data_tabungan['pengambilan']);
            $saldo_baru = $saldo_sebelumnya + $setoran_baru - $pengambilan;
            
            $stmt_update = $conn->prepare("UPDATE tabungan_usipa SET 
                setoran = ?, 
                saldo = ?, 
                tanggal_transaksi = ? 
                WHERE id_tabungan = ?");
            
            if (!$stmt_update) {
                throw new Exception("Prepare update tabungan gagal: " . $conn->error);
            }
            
            $stmt_update->bind_param("ddsi", $setoran_baru, $saldo_baru, $tanggal_setor, $data_tabungan['id_tabungan']);
            
            if (!$stmt_update->execute()) {
                throw new Exception("Error updating tabungan: " . $stmt_update->error);
            }
            
            $stmt_update->close();
        } else {
            // Insert new record
            $saldo_baru = $saldo_sebelumnya + $nominal;
            
            $stmt_insert = $conn->prepare("INSERT INTO tabungan_usipa 
                (id_anggota, tahun, bulan, setoran, pengambilan, saldo, persentase_jasa, jasa, tanggal_transaksi) 
                VALUES (?, ?, ?, ?, 0, ?, 0, 0, ?)");
            
            if (!$stmt_insert) {
                throw new Exception("Prepare insert tabungan gagal: " . $conn->error);
            }
            
            $stmt_insert->bind_param("iiidds", $id_anggota, $tahun, $bulan, $nominal, $saldo_baru, $tanggal_setor);
            
            if (!$stmt_insert->execute()) {
                throw new Exception("Error inserting tabungan: " . $stmt_insert->error);
            }
            
            $stmt_insert->close();
        }
        
        $stmt_cek_tabungan->close();
        
        // 5. Update atau Insert ke tabel dana_unit
        $stmt_cek_dana = $conn->prepare("SELECT id_dana, total_dana FROM dana_unit 
            WHERE id_unit = ? AND tahun = ? AND bulan = ?");
        
        if (!$stmt_cek_dana) {
            throw new Exception("Prepare cek dana_unit gagal: " . $conn->error);
        }
        
        $stmt_cek_dana->bind_param("iii", $id_unit, $tahun, $bulan);
        $stmt_cek_dana->execute();
        $result_dana = $stmt_cek_dana->get_result();
        
        $keterangan_dana = "Simpanan Masuk bulan " . $bulan_nama[$bulan] . " " . $tahun;
        
        if ($result_dana && $result_dana->num_rows > 0) {
            // Update existing record - tambahkan ke total_dana yang sudah ada
            $data_dana = $result_dana->fetch_assoc();
            $total_dana_baru = floatval($data_dana['total_dana']) + $nominal;
            
            $stmt_update_dana = $conn->prepare("UPDATE dana_unit SET 
                total_dana = ?, 
                keterangan = ?,
                tanggal_input = ? 
                WHERE id_dana = ?");
            
            if (!$stmt_update_dana) {
                throw new Exception("Prepare update dana_unit gagal: " . $conn->error);
            }
            
            $stmt_update_dana->bind_param("dssi", $total_dana_baru, $keterangan_dana, $tanggal_setor, $data_dana['id_dana']);
            
            if (!$stmt_update_dana->execute()) {
                throw new Exception("Error updating dana_unit: " . $stmt_update_dana->error);
            }
            
            $stmt_update_dana->close();
        } else {
            // Insert new record
            $stmt_insert_dana = $conn->prepare("INSERT INTO dana_unit 
                (id_unit, tahun, bulan, total_dana, keterangan, tanggal_input) 
                VALUES (?, ?, ?, ?, ?, ?)");
            
            if (!$stmt_insert_dana) {
                throw new Exception("Prepare insert dana_unit gagal: " . $conn->error);
            }
            
            $stmt_insert_dana->bind_param("iiidss", $id_unit, $tahun, $bulan, $nominal, $keterangan_dana, $tanggal_setor);
            
            if (!$stmt_insert_dana->execute()) {
                throw new Exception("Error inserting dana_unit: " . $stmt_insert_dana->error);
            }
            
            $stmt_insert_dana->close();
        }
        
        $stmt_cek_dana->close();
        
        // Commit transaksi
        $conn->commit();
        
        header("Location: input_simpanan_sukarela.php?success=1&nominal=$nominal");
        exit;
        
    } catch (Exception $e) {
        $conn->rollback();
        error_log("Error simpanan sukarela: " . $e->getMessage());
        header("Location: input_simpanan_sukarela.php?error=gagal&msg=" . urlencode($e->getMessage()));
        exit;
    }
}

// Ambil daftar anggota untuk dropdown
$anggota_list = $conn->query("SELECT id_anggota, nama, nip FROM anggota ORDER BY nama ASC");

// Ambil riwayat simpanan sukarela
$riwayat = $conn->query("SELECT 
    s.id_simpanan,
    s.id_anggota,
    a.nama,
    a.nip,
    s.nominal,
    s.tanggal_setor,
    s.keterangan
    FROM simpanan s
    JOIN anggota a ON s.id_anggota = a.id_anggota
    WHERE s.jenis = 'sukarela'
    ORDER BY s.tanggal_setor DESC, s.id_simpanan DESC
    LIMIT 50");
?>

<!DOCTYPE html>
<html lang="id">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Input Simpanan Sukarela - Koperasi</title>
    <script src="https://cdn.tailwindcss.com"></script>
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.0.0/css/all.min.css">
    <script>
        tailwind.config = {
            theme: {
                extend: {
                    colors: {
                        primary: {
                            50: '#f5f7ff',
                            500: '#667eea',
                            600: '#5568d3',
                        },
                        purple: {
                            500: '#764ba2',
                        }
                    }
                }
            }
        }
    </script>
</head>
<body class="bg-gradient-to-br from-green-500 via-teal-500 to-blue-600 min-h-screen">
    
    <!-- Sidebar -->
    <aside class="fixed top-0 left-0 h-screen w-64 bg-white shadow-2xl z-50 overflow-y-auto">
        <div class="p-6 border-b border-gray-200 bg-gradient-to-r from-green-500 to-teal-500">
            <div class="flex items-center space-x-3 mb-4">
                <img src="img/logo.png" alt="Logo" class="w-16 h-16 object-contain bg-white rounded-xl p-2 shadow-lg">
                <div>
                    <h2 class="text-white font-bold text-lg">Koperasi</h2>
                    <p class="text-green-100 text-xs">Unit <?= htmlspecialchars($nama_unit) ?></p>
                </div>
            </div>
            <div class="pt-3 border-t border-green-400">
                <div class="flex items-center space-x-3">
                    <div class="w-10 h-10 bg-white rounded-lg flex items-center justify-center shadow">
                        <i class="fas fa-user-tie text-green-600 text-xl"></i>
                    </div>
                    <div>
                        <h3 class="text-white font-bold text-sm">Kepala Unit <?= htmlspecialchars($nama_unit) ?></h3>
                        <p class="text-green-100 text-xs">Simpanan Sukarela</p>
                    </div>
                </div>
            </div>
        </div>

        <nav class="p-4 space-y-2">
            <a href="dashboard_kepalaunit_toko.php?menu=approval_pinjaman" class="flex items-center space-x-3 p-3 rounded-lg text-gray-700 hover:bg-gray-100">
                <i class="fas fa-hand-holding-usd w-5"></i>
                <span class="font-medium">Approval Pinjaman</span>
            </a>
            
            <a href="potong_gaji.php" class="flex items-center space-x-3 p-3 rounded-lg text-gray-700 hover:bg-gray-100">
                <i class="fas fa-cut w-5"></i>
                <span class="font-medium">Potong Gaji</span>
            </a>
            
            <a href="input_simpanan_sukarela.php" class="flex items-center space-x-3 p-3 rounded-lg bg-gradient-to-r from-green-500 to-teal-500 text-white">
                <i class="fas fa-hand-holding-heart w-5"></i>
                <span class="font-medium">Simpanan Sukarela</span>
            </a>
            
            <a href="dana_unit.php" class="flex items-center space-x-3 p-3 rounded-lg text-gray-700 hover:bg-gray-100">
                <i class="fas fa-money-bill-wave w-5"></i>
                <span class="font-medium">Dana Unit</span>
            </a>

             <a href="konfigurasi_shu.php" class="flex items-center space-x-3 p-3 rounded-lg text-gray-700 hover:bg-gray-100">
                <i class="fas fa-chart-pie w-5"></i>
                <span class="font-medium">Konfigurasi SHU</span>
            </a>

            <div class="border-t border-gray-200 my-4"></div>

            <a href="logout.php" class="flex items-center space-x-3 p-3 rounded-lg text-red-600 hover:bg-red-50">
                <i class="fas fa-sign-out-alt w-5"></i>
                <span class="font-medium">Logout</span>
            </a>
        </nav>
    </aside>

    <!-- Main Content -->
    <main class="ml-64 p-8">
        <div class="max-w-7xl mx-auto">
            
            <!-- Header -->
            <div class="mb-8">
                <h1 class="text-4xl font-bold text-white mb-2">
                    <i class="fas fa-hand-holding-heart mr-3"></i>Input Simpanan Sukarela
                </h1>
                <p class="text-green-100">Input simpanan sukarela anggota koperasi</p>
            </div>

            <?php if (isset($_GET['success'])): ?>
            <div class="mb-6 bg-green-100 border border-green-400 text-green-700 px-4 py-3 rounded-lg shadow">
                <i class="fas fa-check-circle mr-2"></i>
                Berhasil menyimpan data simpanan sukarela sebesar Rp <?= number_format($_GET['nominal'], 0, ',', '.') ?>!
            </div>
            <?php endif; ?>

            <?php if (isset($_GET['error'])): ?>
            <div class="mb-6 bg-red-100 border border-red-400 text-red-700 px-4 py-3 rounded-lg shadow">
                <i class="fas fa-exclamation-circle mr-2"></i>
                <?php if ($_GET['error'] == 'anggota_kosong'): ?>
                    Silakan pilih anggota terlebih dahulu!
                <?php elseif ($_GET['error'] == 'nominal_invalid'): ?>
                    Nominal simpanan harus lebih dari Rp 0!
                <?php elseif ($_GET['error'] == 'tanggal_kosong'): ?>
                    Tanggal setor tidak boleh kosong!
                <?php elseif ($_GET['error'] == 'gagal'): ?>
                    Terjadi kesalahan saat menyimpan data!
                    <?php if (isset($_GET['msg'])): ?>
                        <br><small>Detail: <?= htmlspecialchars($_GET['msg']) ?></small>
                    <?php endif; ?>
                <?php else: ?>
                    Terjadi kesalahan!
                <?php endif; ?>
            </div>
            <?php endif; ?>

            <!-- Form Input Simpanan Sukarela -->
            <div class="bg-white rounded-2xl shadow-2xl overflow-hidden mb-8">
                <div class="bg-gradient-to-r from-green-500 to-teal-500 p-6">
                    <h2 class="text-2xl font-bold text-white">
                        <i class="fas fa-edit mr-3"></i>Form Input Simpanan Sukarela
                    </h2>
                    <p class="text-green-100 text-sm mt-1">Isi form di bawah untuk menambah simpanan sukarela</p>
                </div>
                
                <div class="p-6">
                    <div class="bg-gradient-to-br from-green-50 to-teal-50 border border-green-200 rounded-xl p-6">
                        <form method="POST" id="formSimpanan" onsubmit="return validateForm()">
                            <div class="grid grid-cols-1 md:grid-cols-2 gap-6 mb-6">
                                <div class="md:col-span-2">
                                    <label class="block text-sm font-medium text-gray-700 mb-2">
                                        <i class="fas fa-user mr-1"></i>Pilih Anggota <span class="text-red-500">*</span>
                                    </label>
                                    <select name="id_anggota" id="id_anggota" required class="w-full px-4 py-3 border rounded-lg focus:ring-2 focus:ring-green-500">
                                        <option value="">-- Pilih Anggota --</option>
                                        <?php if ($anggota_list && $anggota_list->num_rows > 0): ?>
                                            <?php while ($anggota = $anggota_list->fetch_assoc()): ?>
                                                <option value="<?= $anggota['id_anggota'] ?>">
                                                    <?= htmlspecialchars($anggota['nama']) ?> (NIP: <?= htmlspecialchars($anggota['nip']) ?>)
                                                </option>
                                            <?php endwhile; ?>
                                        <?php endif; ?>
                                    </select>
                                </div>
                                
                                <div>
                                    <label class="block text-sm font-medium text-gray-700 mb-2">
                                        <i class="fas fa-money-bill-wave mr-1"></i>Nominal Simpanan <span class="text-red-500">*</span>
                                    </label>
                                    <input type="text" name="nominal" id="nominal" required 
                                        placeholder="Contoh: 100000" 
                                        class="w-full px-4 py-3 border rounded-lg focus:ring-2 focus:ring-green-500"
                                        onkeyup="formatRupiah(this)">
                                    <p class="text-xs text-gray-500 mt-1">Masukkan nominal tanpa titik atau koma</p>
                                </div>
                                
                                <div>
                                    <label class="block text-sm font-medium text-gray-700 mb-2">
                                        <i class="fas fa-calendar mr-1"></i>Tanggal Setor <span class="text-red-500">*</span>
                                    </label>
                                    <input type="date" name="tanggal_setor" id="tanggal_setor" required 
                                        value="<?= date('Y-m-d') ?>"
                                        max="<?= date('Y-m-d') ?>"
                                        class="w-full px-4 py-3 border rounded-lg focus:ring-2 focus:ring-green-500">
                                </div>
                                
                                <div class="md:col-span-2">
                                    <label class="block text-sm font-medium text-gray-700 mb-2">
                                        <i class="fas fa-sticky-note mr-1"></i>Keterangan (Opsional)
                                    </label>
                                    <textarea name="keterangan" id="keterangan" rows="3"
                                        placeholder="Contoh: Simpanan tambahan untuk keperluan..."
                                        class="w-full px-4 py-3 border rounded-lg focus:ring-2 focus:ring-green-500"></textarea>
                                    <p class="text-xs text-gray-500 mt-1">Kosongkan jika tidak ada keterangan khusus</p>
                                </div>
                            </div>
                            
                            <div class="bg-blue-50 border border-blue-200 rounded-lg p-4 mb-6">
                                <h4 class="font-semibold text-blue-800 mb-2">
                                    <i class="fas fa-info-circle mr-2"></i>Informasi Penting:
                                </h4>
                                <ul class="text-sm text-blue-700 space-y-1">
                                    <li>• Simpanan sukarela adalah simpanan yang dilakukan secara sukarela oleh anggota</li>
                                    <li>• Nominal dapat disesuaikan dengan kemampuan anggota</li>
                                    <li>• Data akan otomatis masuk ke tabel simpanan, tabungan_usipa, dan dana_unit</li>
                                    <li>• Saldo tabungan dan dana unit akan bertambah sesuai nominal yang diinput</li>
                                    <li>• Tanggal setor tidak boleh melebihi tanggal hari ini</li>
                                </ul>
                            </div>
                            
                            <div class="flex gap-4">
                                <button type="submit" name="simpan_sukarela" 
                                    class="flex-1 bg-gradient-to-r from-green-500 to-teal-500 text-white px-6 py-4 rounded-lg font-semibold hover:shadow-lg transition text-lg">
                                    <i class="fas fa-save mr-2"></i>Simpan Data
                                </button>
                                <button type="reset" onclick="resetForm()"
                                    class="px-6 py-4 bg-gray-200 text-gray-700 rounded-lg font-semibold hover:bg-gray-300 transition">
                                    <i class="fas fa-redo mr-2"></i>Reset
                                </button>
                            </div>
                        </form>
                    </div>
                </div>
            </div>

            <!-- Riwayat Simpanan Sukarela -->
            <div class="bg-white rounded-2xl shadow-2xl overflow-hidden">
                <div class="bg-gradient-to-r from-purple-500 to-indigo-500 p-6">
                    <h2 class="text-2xl font-bold text-white">
                        <i class="fas fa-history mr-3"></i>Riwayat Simpanan Sukarela
                    </h2>
                    <p class="text-purple-100 text-sm mt-1">50 transaksi terakhir</p>
                </div>
                
                <div class="p-6">
                    <div class="overflow-x-auto">
                        <table class="w-full">
                            <thead>
                                <tr class="bg-gray-50 border-b-2 border-purple-500">
                                    <th class="px-6 py-4 text-left text-xs font-bold text-gray-700 uppercase">Tanggal</th>
                                    <th class="px-6 py-4 text-left text-xs font-bold text-gray-700 uppercase">NIP</th>
                                    <th class="px-6 py-4 text-left text-xs font-bold text-gray-700 uppercase">Nama Anggota</th>
                                    <th class="px-6 py-4 text-right text-xs font-bold text-gray-700 uppercase">Nominal</th>
                                    <th class="px-6 py-4 text-left text-xs font-bold text-gray-700 uppercase">Keterangan</th>
                                </tr>
                            </thead>
                            <tbody class="divide-y divide-gray-200">
                                <?php if ($riwayat && $riwayat->num_rows > 0): ?>
                                    <?php while ($row = $riwayat->fetch_assoc()): ?>
                                    <tr class="hover:bg-gray-50">
                                        <td class="px-6 py-4"><?= date('d/m/Y', strtotime($row['tanggal_setor'])) ?></td>
                                        <td class="px-6 py-4 font-medium"><?= htmlspecialchars($row['nip']) ?></td>
                                        <td class="px-6 py-4 font-semibold"><?= htmlspecialchars($row['nama']) ?></td>
                                        <td class="px-6 py-4 text-right">
                                            <span class="text-lg font-bold text-green-600">Rp <?= number_format($row['nominal'], 0, ',', '.') ?></span>
                                        </td>
                                        <td class="px-6 py-4 text-sm text-gray-600"><?= htmlspecialchars($row['keterangan']) ?></td>
                                    </tr>
                                    <?php endwhile; ?>
                                <?php else: ?>
                                    <tr>
                                        <td colspan="5" class="px-6 py-12 text-center text-gray-500">
                                            <i class="fas fa-inbox text-6xl mb-4 text-gray-300"></i>
                                            <p>Belum ada riwayat simpanan sukarela</p>
                                        </td>
                                    </tr>
                                <?php endif; ?>
                            </tbody>
                        </table>
                    </div>
                </div>
            </div>

        </div>
    </main>

    <script>
        function formatRupiah(input) {
            let value = input.value.replace(/[^0-9]/g, '');
            input.value = value;
        }
        
        function validateForm() {
            const idAnggota = document.getElementById('id_anggota').value;
            const nominal = document.getElementById('nominal').value;
            const tanggal = document.getElementById('tanggal_setor').value;
            
            if (!idAnggota) {
                alert('Silakan pilih anggota terlebih dahulu!');
                return false;
            }
            
            if (!nominal || parseFloat(nominal.replace(/[^0-9]/g, '')) <= 0) {
                alert('Nominal simpanan harus lebih dari Rp 0!');
                return false;
            }
            
            if (!tanggal) {
                alert('Tanggal setor tidak boleh kosong!');
                return false;
            }
            
            // Validasi tanggal tidak boleh melebihi hari ini
            const today = new Date();
            const selectedDate = new Date(tanggal);
            
            if (selectedDate > today) {
                alert('Tanggal setor tidak boleh melebihi tanggal hari ini!');
                return false;
            }
            
            return confirm('Anda yakin akan menyimpan data simpanan sukarela ini?');
        }
        
        function resetForm() {
            document.getElementById('tanggal_setor').value = '<?= date('Y-m-d') ?>';
        }
    </script>

</body>
</html>