<?php
session_start();
include 'config.php';

// Tambahkan error reporting untuk debugging (hapus di production)
error_reporting(E_ALL);
ini_set('display_errors', 1);

// Cek apakah user login dan role = Kepala Unit Toko (id_role = 7)
if (!isset($_SESSION['role']) || $_SESSION['id_role'] != 7) {
    header("Location: index.php");
    exit;
}

$id_unit = 1; 
$nama_unit = "Simpan Pinjam";

// Proses simpan konfigurasi SHU
if (isset($_POST['simpan_konfigurasi_shu'])) {
    $tahun = intval($_POST['tahun_shu']);
    $persentase_simpan_pinjam = floatval($_POST['persentase_simpan_pinjam']);
    $persentase_defmart = floatval($_POST['persentase_defmart']);
    $persentase_pujasera = floatval($_POST['persentase_pujasera']);
    $suku_bunga = floatval($_POST['suku_bunga']);
    
    // Validasi total persentase = 100%
    $total_persentase = $persentase_simpan_pinjam + $persentase_defmart + $persentase_pujasera;
    if (abs($total_persentase - 100) > 0.01) {
        header("Location: konfigurasi_shu.php?error=persentase&total=" . $total_persentase);
        exit;
    }
    
    // Cek apakah sudah ada konfigurasi untuk tahun tersebut
    $stmt_check = $conn->prepare("SELECT id_konfigurasi FROM konfigurasi_shu WHERE tahun = ?");
    $stmt_check->bind_param("i", $tahun);
    $stmt_check->execute();
    $check_result = $stmt_check->get_result();
    
    if ($check_result->num_rows > 0) {
        // Update konfigurasi yang sudah ada
        $stmt_update = $conn->prepare("UPDATE konfigurasi_shu SET 
            persentase_simpan_pinjam = ?, 
            persentase_defmart = ?, 
            persentase_pujasera = ?, 
            suku_bunga = ?, 
            tanggal_ditetapkan = CURDATE() 
            WHERE tahun = ?");
        $stmt_update->bind_param("ddddi", $persentase_simpan_pinjam, $persentase_defmart, $persentase_pujasera, $suku_bunga, $tahun);
        $stmt_update->execute();
    } else {
        // Insert konfigurasi baru
        $stmt_insert = $conn->prepare("INSERT INTO konfigurasi_shu 
            (tahun, persentase_simpan_pinjam, persentase_defmart, persentase_pujasera, suku_bunga, tanggal_ditetapkan) 
            VALUES (?, ?, ?, ?, ?, CURDATE())");
        $stmt_insert->bind_param("idddd", $tahun, $persentase_simpan_pinjam, $persentase_defmart, $persentase_pujasera, $suku_bunga);
        $stmt_insert->execute();
    }
    
    header("Location: konfigurasi_shu.php?success=config");
    exit;
}

// Proses perhitungan SHU USIPA
if (isset($_POST['hitung_shu_usipa'])) {
    $tahun = intval($_POST['tahun_hitung']);
    $id_user = $_SESSION['id_user'];
    
    // Ambil konfigurasi SHU untuk tahun tersebut
    $stmt_config = $conn->prepare("SELECT * FROM konfigurasi_shu WHERE tahun = ?");
    $stmt_config->bind_param("i", $tahun);
    $stmt_config->execute();
    $config_result = $stmt_config->get_result();
    
    if ($config_result->num_rows == 0) {
        header("Location: konfigurasi_shu.php?error=no_config");
        exit;
    }
    
    $config = $config_result->fetch_assoc();
    $suku_bunga = floatval($config['suku_bunga']);
    
    // Hapus perhitungan SHU sebelumnya untuk tahun yang sama
    $stmt_delete = $conn->prepare("DELETE FROM shu_pembagian WHERE tahun = ?");
    $stmt_delete->bind_param("i", $tahun);
    $stmt_delete->execute();
    
    // Ambil semua anggota yang memiliki tabungan di tahun tersebut
    $stmt_anggota = $conn->prepare("
        SELECT DISTINCT t.id_anggota, a.nama, a.nip
        FROM tabungan_usipa t
        JOIN anggota a ON t.id_anggota = a.id_anggota
        WHERE t.tahun = ?
        ORDER BY a.nama
    ");
    
    $stmt_anggota->bind_param("i", $tahun);
    $stmt_anggota->execute();
    $result_anggota = $stmt_anggota->get_result();
    
    if ($result_anggota && $result_anggota->num_rows > 0) {
        $conn->begin_transaction();
        
        try {
            $stmt_insert_shu = $conn->prepare("INSERT INTO shu_pembagian 
                (id_anggota, tahun, shu_toko, shu_usipa, shu_jasa_umum, total_shu) 
                VALUES (?, ?, 0, ?, 0, ?)");
            
            $total_anggota = 0;
            $total_shu_dibagikan = 0;
            
            while ($anggota = $result_anggota->fetch_assoc()) {
                $id_anggota = $anggota['id_anggota'];
                
                // Ambil saldo per bulan untuk anggota ini
                $stmt_saldo = $conn->prepare("
                    SELECT bulan, saldo
                    FROM tabungan_usipa
                    WHERE id_anggota = ? AND tahun = ?
                    ORDER BY bulan ASC
                ");
                $stmt_saldo->bind_param("ii", $id_anggota, $tahun);
                $stmt_saldo->execute();
                $result_saldo = $stmt_saldo->get_result();
                
                // Jumlahkan saldo dari bulan 1-12
                $total_saldo_tahunan = 0;
                $detail_bulan = [];
                
                while ($row = $result_saldo->fetch_assoc()) {
                    $bulan = intval($row['bulan']);
                    $saldo = floatval($row['saldo']);
                    $total_saldo_tahunan += $saldo;
                    $detail_bulan[$bulan] = $saldo;
                }
                
                // Perhitungan SHU USIPA: Total Saldo Tahunan × (Suku Bunga / 100)
                $shu_usipa = $total_saldo_tahunan * ($suku_bunga / 100);
                $total_shu = $shu_usipa;
                
                // Simpan hasil perhitungan SHU
                $stmt_insert_shu->bind_param("iidd", 
                    $id_anggota, 
                    $tahun, 
                    $shu_usipa, 
                    $total_shu
                );
                
                if (!$stmt_insert_shu->execute()) {
                    throw new Exception("Error inserting SHU untuk anggota " . $anggota['nama'] . ": " . $stmt_insert_shu->error);
                }
                
                $total_anggota++;
                $total_shu_dibagikan += $shu_usipa;
                
                $stmt_saldo->close();
            }
            
            $conn->commit();
            $stmt_insert_shu->close();
            
            header("Location: konfigurasi_shu.php?success=hitung&anggota=$total_anggota&total=" . number_format($total_shu_dibagikan, 0, ',', '.'));
            exit;
            
        } catch (Exception $e) {
            $conn->rollback();
            error_log("Error hitung SHU: " . $e->getMessage());
            header("Location: konfigurasi_shu.php?error=gagal&msg=" . urlencode($e->getMessage()));
            exit;
        }
    } else {
        header("Location: konfigurasi_shu.php?error=no_data");
        exit;
    }
}

$tahun_shu_view = isset($_GET['tahun_shu_view']) ? intval($_GET['tahun_shu_view']) : date('Y');

// Array nama bulan
$bulan_nama = ['', 'Januari', 'Februari', 'Maret', 'April', 'Mei', 'Juni', 
              'Juli', 'Agustus', 'September', 'Oktober', 'November', 'Desember'];
?>

<!DOCTYPE html>
<html lang="id">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Konfigurasi SHU - Koperasi</title>
    <script src="https://cdn.tailwindcss.com"></script>
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.0.0/css/all.min.css">
    <script>
        tailwind.config = {
            theme: {
                extend: {
                    colors: {
                        primary: {
                            50: '#f5f7ff',
                            500: '#667eea',
                            600: '#5568d3',
                        },
                        purple: {
                            500: '#764ba2',
                        }
                    }
                }
            }
        }
    </script>
</head>
<body class="bg-gradient-to-br from-primary-500 via-purple-500 to-purple-600 min-h-screen">
    
    <!-- Sidebar -->
    <aside class="fixed top-0 left-0 h-screen w-64 bg-white shadow-2xl z-50 overflow-y-auto">
        <div class="p-6 border-b border-gray-200 bg-gradient-to-r from-primary-500 to-purple-500">
            <div class="flex items-center space-x-3 mb-4">
                <img src="img/logo.png" alt="Logo" class="w-16 h-16 object-contain bg-white rounded-xl p-2 shadow-lg">
                <div>
                    <h2 class="text-white font-bold text-lg">Koperasi</h2>
                    <p class="text-primary-100 text-xs">Unit <?= htmlspecialchars($nama_unit) ?></p>
                </div>
            </div>
            <div class="pt-3 border-t border-primary-400">
                <div class="flex items-center space-x-3">
                    <div class="w-10 h-10 bg-white rounded-lg flex items-center justify-center shadow">
                        <i class="fas fa-user-tie text-primary-600 text-xl"></i>
                    </div>
                    <div>
                        <h3 class="text-white font-bold text-sm">Kepala Unit <?= htmlspecialchars($nama_unit) ?></h3>
                        <p class="text-primary-100 text-xs">Dashboard</p>
                    </div>
                </div>
            </div>
        </div>

        <nav class="p-4 space-y-2">
            <a href="dashboard_kepalaunit_toko.php?menu=approval_pinjaman" class="flex items-center space-x-3 p-3 rounded-lg text-gray-700 hover:bg-gray-100">
                <i class="fas fa-hand-holding-usd w-5"></i>
                <span class="font-medium">Approval Pinjaman</span>
            </a>
            
            <a href="potong_gaji.php" class="flex items-center space-x-3 p-3 rounded-lg text-gray-700 hover:bg-gray-100">
                <i class="fas fa-cut w-5"></i>
                <span class="font-medium">Potong Gaji</span>
            </a>
            
            <a href="input_simpanan_sukarela.php" class="flex items-center space-x-3 p-3 rounded-lg text-gray-700 hover:bg-gray-100">
                <i class="fas fa-hand-holding-heart w-5"></i>
                <span class="font-medium">Simpanan Sukarela</span>
            </a>
            
            <a href="dana_unit.php" class="flex items-center space-x-3 p-3 rounded-lg text-gray-700 hover:bg-gray-100">
                <i class="fas fa-money-bill-wave w-5"></i>
                <span class="font-medium">Dana Unit</span>
            </a>

            <a href="konfigurasi_shu.php" class="flex items-center space-x-3 p-3 rounded-lg bg-gradient-to-r from-primary-500 to-purple-500 text-white">
                <i class="fas fa-chart-pie w-5"></i>
                <span class="font-medium">Konfigurasi SHU</span>
            </a>

            <div class="border-t border-gray-200 my-4"></div>

            <a href="logout.php" class="flex items-center space-x-3 p-3 rounded-lg text-red-600 hover:bg-red-50">
                <i class="fas fa-sign-out-alt w-5"></i>
                <span class="font-medium">Logout</span>
            </a>
        </nav>
    </aside>

    <!-- Main Content -->
    <main class="ml-64 p-8">
        <div class="max-w-7xl mx-auto">
            
            <!-- Header -->
            <div class="mb-8">
                <h1 class="text-4xl font-bold text-white mb-2">
                    Konfigurasi SHU USIPA
                </h1>
                <p class="text-primary-100">Kelola konfigurasi dan perhitungan Sisa Hasil Usaha (SHU)</p>
            </div>

            <?php if (isset($_GET['success'])): ?>
            <div class="mb-6 bg-green-100 border border-green-400 text-green-700 px-4 py-3 rounded-lg shadow">
                <i class="fas fa-check-circle mr-2"></i>
                <?php if ($_GET['success'] == 'config'): ?>
                    Konfigurasi SHU berhasil disimpan!
                <?php elseif ($_GET['success'] == 'hitung'): ?>
                    Perhitungan SHU USIPA berhasil dilakukan!
                    <?php if (isset($_GET['anggota']) && isset($_GET['total'])): ?>
                        <br><small>Total <?= $_GET['anggota'] ?> anggota dengan total SHU: Rp <?= $_GET['total'] ?></small>
                    <?php endif; ?>
                <?php endif; ?>
            </div>
            <?php endif; ?>

            <?php if (isset($_GET['error'])): ?>
            <div class="mb-6 bg-red-100 border border-red-400 text-red-700 px-4 py-3 rounded-lg shadow">
                <i class="fas fa-exclamation-circle mr-2"></i>
                <?php if ($_GET['error'] == 'no_config'): ?>
                    Belum ada konfigurasi SHU untuk tahun tersebut! Silakan buat konfigurasi terlebih dahulu.
                <?php elseif ($_GET['error'] == 'no_data'): ?>
                    Tidak ada data tabungan untuk tahun tersebut!
                <?php elseif ($_GET['error'] == 'persentase'): ?>
                    Total persentase harus 100%! Saat ini: <?= isset($_GET['total']) ? $_GET['total'] : '0' ?>%
                <?php elseif ($_GET['error'] == 'gagal'): ?>
                    Terjadi kesalahan saat memproses data!
                    <?php if (isset($_GET['msg'])): ?>
                        <br><small>Detail: <?= htmlspecialchars($_GET['msg']) ?></small>
                    <?php endif; ?>
                <?php endif; ?>
            </div>
            <?php endif; ?>

            <div class="space-y-8">
                <!-- Konfigurasi SHU -->
                <div class="bg-white rounded-2xl shadow-2xl overflow-hidden">
                    <div class="bg-gradient-to-r from-blue-500 to-indigo-500 p-6">
                        <h2 class="text-2xl font-bold text-white">
                            <i class="fas fa-cogs mr-3"></i>Konfigurasi SHU USIPA
                        </h2>
                        <p class="text-blue-100 text-sm mt-1">Atur persentase dan suku bunga untuk perhitungan SHU</p>
                    </div>
                    
                    <div class="p-6">
                        <form method="POST" class="space-y-6" id="formKonfigurasi">
                            <div class="grid grid-cols-1 md:grid-cols-2 gap-6">
                                <div>
                                    <label class="block text-sm font-medium text-gray-700 mb-2">Tahun</label>
                                    <select name="tahun_shu" required class="w-full px-4 py-3 border rounded-lg focus:ring-2 focus:ring-blue-500">
                                        <?php for($y = date('Y'); $y >= 2020; $y--): ?>
                                        <option value="<?= $y ?>" <?= $y == date('Y') ? 'selected' : '' ?>><?= $y ?></option>
                                        <?php endfor; ?>
                                    </select>
                                </div>
                                
                                <div>
                                    <label class="block text-sm font-medium text-gray-700 mb-2">Suku Bunga USIPA (%)</label>
                                    <input type="number" name="suku_bunga" step="0.01" min="0" max="100" required 
                                           class="w-full px-4 py-3 border rounded-lg focus:ring-2 focus:ring-blue-500" 
                                           placeholder="contoh: 2.5">
                                    <p class="text-xs text-gray-500 mt-1">Suku bunga untuk menghitung SHU USIPA</p>
                                </div>
                            </div>
                            
                            <div class="grid grid-cols-1 md:grid-cols-3 gap-6">
                                <div>
                                    <label class="block text-sm font-medium text-gray-700 mb-2">% Simpan Pinjam</label>
                                    <input type="number" name="persentase_simpan_pinjam" step="0.01" min="0" max="100" required 
                                           class="w-full px-4 py-3 border rounded-lg focus:ring-2 focus:ring-blue-500 persentase-input" 
                                           placeholder="contoh: 40">
                                </div>
                                
                                <div>
                                    <label class="block text-sm font-medium text-gray-700 mb-2">% Defmart</label>
                                    <input type="number" name="persentase_defmart" step="0.01" min="0" max="100" required 
                                           class="w-full px-4 py-3 border rounded-lg focus:ring-2 focus:ring-blue-500 persentase-input" 
                                           placeholder="contoh: 30">
                                </div>
                                
                                <div>
                                    <label class="block text-sm font-medium text-gray-700 mb-2">% Pujasera</label>
                                    <input type="number" name="persentase_pujasera" step="0.01" min="0" max="100" required 
                                           class="w-full px-4 py-3 border rounded-lg focus:ring-2 focus:ring-blue-500 persentase-input" 
                                           placeholder="contoh: 30">
                                </div>
                            </div>

                            <div id="totalPersentase" class="bg-blue-50 border border-blue-200 rounded-lg p-4">
                                <p class="text-sm text-blue-800 font-semibold">
                                    <i class="fas fa-calculator mr-2"></i>
                                    Total Persentase: <span id="totalPersen" class="text-lg">0</span>%
                                </p>
                            </div>
                            
                            <div class="bg-yellow-50 border border-yellow-200 rounded-lg p-4">
                                <p class="text-sm text-yellow-800 font-semibold mb-2">
                                    <i class="fas fa-info-circle mr-2"></i>
                                    Formula Perhitungan SHU USIPA:
                                </p>
                                <div class="space-y-2">
                                    <p class="text-sm text-yellow-700">
                                        <strong>Langkah 1:</strong> Jumlahkan saldo tabungan dari Bulan 1 hingga Bulan 12
                                    </p>
                                    <p class="text-sm text-yellow-700 ml-4">
                                        Total Saldo Tahunan = Saldo Bulan 1 + Saldo Bulan 2 + ... + Saldo Bulan 12
                                    </p>
                                    <p class="text-sm text-yellow-700">
                                        <strong>Langkah 2:</strong> Hitung SHU berdasarkan suku bunga
                                    </p>
                                    <p class="text-sm text-yellow-700 ml-4">
                                        SHU USIPA = Total Saldo Tahunan × (Suku Bunga ÷ 100)
                                    </p>
                                </div>
                                <p class="text-xs text-yellow-600 mt-3 italic">
                                    * Contoh: Jika total saldo tahunan Rp 12.000.000 dan suku bunga 2.5%, maka SHU = Rp 12.000.000 × 0.025 = Rp 300.000
                                </p>
                            </div>
                            
                            <button type="submit" name="simpan_konfigurasi_shu" class="w-full bg-gradient-to-r from-blue-500 to-indigo-500 text-white px-6 py-4 rounded-lg font-semibold hover:shadow-lg transition">
                                <i class="fas fa-save mr-2"></i>Simpan Konfigurasi
                            </button>
                        </form>
                    </div>
                </div>

                <!-- Hitung SHU USIPA -->
                <div class="bg-white rounded-2xl shadow-2xl overflow-hidden">
                    <div class="bg-gradient-to-r from-green-500 to-emerald-500 p-6">
                        <h2 class="text-2xl font-bold text-white">
                            <i class="fas fa-calculator mr-3"></i>Hitung SHU USIPA
                        </h2>
                        <p class="text-green-100 text-sm mt-1">Proses perhitungan SHU berdasarkan konfigurasi</p>
                    </div>
                    
                    <div class="p-6">
                        <form method="POST" onsubmit="return confirm('Yakin akan menghitung ulang SHU?\n\nProses ini akan:\n1. Menghapus data SHU sebelumnya untuk tahun ini\n2. Menjumlahkan saldo bulan 1-12 untuk setiap anggota\n3. Menghitung SHU berdasarkan konfigurasi yang tersimpan\n\nLanjutkan?')">
                            <div class="mb-6">
                                <label class="block text-sm font-medium text-gray-700 mb-2">Tahun</label>
                                <select name="tahun_hitung" required class="px-4 py-3 border rounded-lg focus:ring-2 focus:ring-green-500">
                                    <?php for($y = date('Y'); $y >= 2020; $y--): ?>
                                    <option value="<?= $y ?>" <?= $y == date('Y') ? 'selected' : '' ?>><?= $y ?></option>
                                    <?php endfor; ?>
                                </select>
                            </div>

                            <div class="bg-green-50 border border-green-200 rounded-lg p-4 mb-6">
                                <p class="text-sm text-green-800 font-semibold mb-2">
                                    <i class="fas fa-list-check mr-2"></i>
                                    Proses Perhitungan:
                                </p>
                                <ol class="text-sm text-green-700 space-y-1 ml-4 list-decimal">
                                    <li>Mengambil konfigurasi SHU untuk tahun yang dipilih</li>
                                    <li>Menjumlahkan saldo tabungan setiap anggota dari bulan 1-12</li>
                                    <li>Menghitung SHU = Total Saldo Tahunan × (Suku Bunga ÷ 100)</li>
                                    <li>Menyimpan hasil perhitungan ke database</li>
                                </ol>
                            </div>
                            
                            <button type="submit" name="hitung_shu_usipa" class="bg-gradient-to-r from-green-500 to-emerald-500 text-white px-6 py-3 rounded-lg font-semibold hover:shadow-lg transition">
                                <i class="fas fa-play mr-2"></i>Hitung SHU
                            </button>
                        </form>
                    </div>
                </div>

                <!-- Data SHU -->
                <div class="bg-white rounded-2xl shadow-2xl overflow-hidden">
                    <div class="bg-gradient-to-r from-purple-500 to-pink-500 p-6">
                        <h2 class="text-2xl font-bold text-white">
                            <i class="fas fa-chart-pie mr-3"></i>Data SHU USIPA
                        </h2>
                        <p class="text-purple-100 text-sm mt-1">Hasil perhitungan SHU per anggota</p>
                    </div>
                    
                    <div class="p-6">
                        <div class="mb-6">
                            <label class="block text-sm font-medium text-gray-700 mb-2">Filter Tahun</label>
                            <select onchange="window.location.href='?tahun_shu_view='+this.value" class="px-4 py-2 border rounded-lg focus:ring-2 focus:ring-purple-500">
                                <?php for($y = date('Y'); $y >= 2020; $y--): ?>
                                <option value="<?= $y ?>" <?= $tahun_shu_view == $y ? 'selected' : '' ?>><?= $y ?></option>
                                <?php endfor; ?>
                            </select>
                        </div>

                        <?php
                        // Ambil total SHU dan jumlah anggota
                        $stmt_total_shu = $conn->prepare("
                            SELECT 
                                COUNT(*) as jumlah_anggota,
                                SUM(shu_usipa) as total_shu 
                            FROM shu_pembagian 
                            WHERE tahun = ?
                        ");
                        $stmt_total_shu->bind_param("i", $tahun_shu_view);
                        $stmt_total_shu->execute();
                        $total_shu_result = $stmt_total_shu->get_result();
                        $total_data = $total_shu_result->fetch_assoc();
                        $total_shu = $total_data['total_shu'] ?? 0;
                        $jumlah_anggota = $total_data['jumlah_anggota'] ?? 0;
                        ?>

                        <?php if ($total_shu > 0): ?>
                        <div class="grid grid-cols-1 md:grid-cols-3 gap-6 mb-8">
                            <div class="bg-gradient-to-br from-purple-50 to-pink-50 border border-purple-200 rounded-xl p-6">
                                <h3 class="text-sm font-semibold text-gray-600 mb-2">Total SHU USIPA</h3>
                                <p class="text-3xl font-bold text-purple-600">Rp <?= number_format($total_shu, 0, ',', '.') ?></p>
                            </div>
                            <div class="bg-gradient-to-br from-blue-50 to-indigo-50 border border-blue-200 rounded-xl p-6">
                                <h3 class="text-sm font-semibold text-gray-600 mb-2">Jumlah Anggota</h3>
                                <p class="text-3xl font-bold text-blue-600"><?= number_format($jumlah_anggota) ?> Orang</p>
                            </div>
                            <div class="bg-gradient-to-br from-green-50 to-emerald-50 border border-green-200 rounded-xl p-6">
                                <h3 class="text-sm font-semibold text-gray-600 mb-2">Rata-rata SHU</h3>
                                <p class="text-3xl font-bold text-green-600">Rp <?= $jumlah_anggota > 0 ? number_format($total_shu / $jumlah_anggota, 0, ',', '.') : '0' ?></p>
                            </div>
                        </div>
                        <?php endif; ?>

                        <div class="overflow-x-auto">
                            <table class="w-full">
                                <thead>
                                    <tr class="bg-gray-50 border-b-2 border-purple-500">
                                        <th class="px-6 py-4 text-left text-xs font-bold text-gray-700 uppercase">No</th>
                                        <th class="px-6 py-4 text-left text-xs font-bold text-gray-700 uppercase">Anggota</th>
                                        <th class="px-6 py-4 text-right text-xs font-bold text-gray-700 uppercase">Total Saldo Tahunan</th>
                                        <th class="px-6 py-4 text-center text-xs font-bold text-gray-700 uppercase">Jumlah Bulan</th>
                                        <th class="px-6 py-4 text-right text-xs font-bold text-gray-700 uppercase">SHU USIPA</th>
                                        <th class="px-6 py-4 text-center text-xs font-bold text-gray-700 uppercase">Detail</th>
                                    </tr>
                                </thead>
                                <tbody class="divide-y divide-gray-200">
                                    <?php
                                    $stmt_shu = $conn->prepare("
                                        SELECT s.*, a.nama, a.nip,
                                        (SELECT SUM(t.saldo) FROM tabungan_usipa t 
                                         WHERE t.id_anggota = s.id_anggota AND t.tahun = s.tahun) as total_saldo_tahunan,
                                        (SELECT COUNT(DISTINCT t.bulan) FROM tabungan_usipa t 
                                         WHERE t.id_anggota = s.id_anggota AND t.tahun = s.tahun) as jumlah_bulan
                                        FROM shu_pembagian s
                                        JOIN anggota a ON s.id_anggota = a.id_anggota
                                        WHERE s.tahun = ?
                                        ORDER BY s.shu_usipa DESC
                                    ");
                                    $stmt_shu->bind_param("i", $tahun_shu_view);
                                    $stmt_shu->execute();
                                    $shu_result = $stmt_shu->get_result();
                                    
                                    if ($shu_result && $shu_result->num_rows > 0):
                                        $no = 1;
                                        while ($shu = $shu_result->fetch_assoc()):
                                    ?>
                                    <tr class="hover:bg-gray-50">
                                        <td class="px-6 py-4 text-center font-semibold text-gray-600"><?= $no++ ?></td>
                                        <td class="px-6 py-4">
                                            <div class="font-semibold text-gray-800"><?= htmlspecialchars($shu['nama']) ?></div>
                                            <div class="text-xs text-gray-500">NIP: <?= htmlspecialchars($shu['nip']) ?></div>
                                        </td>
                                        <td class="px-6 py-4 text-right">
                                            <span class="text-sm font-semibold text-blue-600">Rp <?= number_format($shu['total_saldo_tahunan'], 0, ',', '.') ?></span>
                                        </td>
                                        <td class="px-6 py-4 text-center">
                                            <span class="px-3 py-1 bg-blue-100 text-blue-800 rounded-full text-xs font-semibold">
                                                <?= $shu['jumlah_bulan'] ?> bulan
                                            </span>
                                        </td>
                                        <td class="px-6 py-4 text-right">
                                            <span class="text-lg font-bold text-purple-600">Rp <?= number_format($shu['shu_usipa'], 0, ',', '.') ?></span>
                                        </td>
                                        <td class="px-6 py-4 text-center">
                                            <button onclick="lihatDetail(<?= $shu['id_anggota'] ?>, <?= $tahun_shu_view ?>, '<?= htmlspecialchars($shu['nama']) ?>')" 
                                                    class="text-blue-600 hover:text-blue-800 font-semibold text-sm">
                                                <i class="fas fa-eye mr-1"></i>Lihat
                                            </button>
                                        </td>
                                    </tr>
                                    <?php 
                                        endwhile;
                                    else:
                                    ?>
                                    <tr>
                                        <td colspan="6" class="px-6 py-12 text-center text-gray-500">
                                            <i class="fas fa-inbox text-6xl mb-4 text-gray-300"></i>
                                            <p class="text-lg font-semibold">Belum ada data SHU untuk tahun <?= $tahun_shu_view ?></p>
                                            <p class="text-sm mt-2">Silakan lakukan perhitungan SHU terlebih dahulu</p>
                                        </td>
                                    </tr>
                                    <?php endif; ?>
                                </tbody>
                            </table>
                        </div>
                    </div>
                </div>
            </div>

        </div>
    </main>

    <!-- Modal Detail Saldo per Bulan -->
    <div id="modalDetail" class="hidden fixed inset-0 bg-black bg-opacity-50 z-50 flex items-center justify-center p-4">
        <div class="bg-white rounded-2xl shadow-2xl max-w-4xl w-full max-h-[90vh] overflow-hidden">
            <div class="bg-gradient-to-r from-purple-500 to-pink-500 p-6">
                <div class="flex justify-between items-center">
                    <div>
                        <h3 class="text-2xl font-bold text-white" id="modalNama">Detail Saldo Bulanan</h3>
                        <p class="text-purple-100 text-sm mt-1" id="modalTahun">Tahun 2025</p>
                    </div>
                    <button onclick="tutupModal()" class="text-white hover:text-gray-200">
                        <i class="fas fa-times text-2xl"></i>
                    </button>
                </div>
            </div>
            <div class="p-6 overflow-y-auto max-h-[70vh]" id="modalContent">
                <!-- Content akan diisi dengan JavaScript -->
            </div>
        </div>
    </div>

    <script>
        // Hitung total persentase secara real-time
        document.querySelectorAll('.persentase-input').forEach(input => {
            input.addEventListener('input', hitungTotalPersentase);
        });

        function hitungTotalPersentase() {
            const p1 = parseFloat(document.querySelector('input[name="persentase_simpan_pinjam"]').value) || 0;
            const p2 = parseFloat(document.querySelector('input[name="persentase_defmart"]').value) || 0;
            const p3 = parseFloat(document.querySelector('input[name="persentase_pujasera"]').value) || 0;
            const total = p1 + p2 + p3;
            
            document.getElementById('totalPersen').textContent = total.toFixed(2);
            
            const div = document.getElementById('totalPersentase');
            if (Math.abs(total - 100) < 0.01) {
                div.className = 'bg-green-50 border border-green-200 rounded-lg p-4';
                div.querySelector('p').className = 'text-sm text-green-800 font-semibold';
            } else {
                div.className = 'bg-red-50 border border-red-200 rounded-lg p-4';
                div.querySelector('p').className = 'text-sm text-red-800 font-semibold';
            }
        }

        // Panggil saat halaman load
        hitungTotalPersentase();

        // Validasi form sebelum submit
        document.getElementById('formKonfigurasi').addEventListener('submit', function(e) {
            const total = parseFloat(document.getElementById('totalPersen').textContent);
            if (Math.abs(total - 100) > 0.01) {
                e.preventDefault();
                alert('Total persentase harus 100%!\nSaat ini: ' + total + '%');
            }
        });

        // Fungsi untuk menampilkan detail saldo per bulan
        function lihatDetail(idAnggota, tahun, nama) {
            document.getElementById('modalNama').textContent = 'Detail Saldo Bulanan - ' + nama;
            document.getElementById('modalTahun').textContent = 'Tahun ' + tahun;
            
            // Ambil data via AJAX
            fetch('get_detail_saldo.php?id_anggota=' + idAnggota + '&tahun=' + tahun)
                .then(response => response.json())
                .then(data => {
                    let html = '<div class="space-y-4">';
                    
                    if (data.success) {
                        const bulanNama = ['', 'Januari', 'Februari', 'Maret', 'April', 'Mei', 'Juni', 
                                          'Juli', 'Agustus', 'September', 'Oktober', 'November', 'Desember'];
                        
                        html += '<div class="grid grid-cols-1 md:grid-cols-2 gap-4">';
                        
                        for (let i = 1; i <= 12; i++) {
                            const item = data.data.find(d => d.bulan == i);
                            const saldo = item ? parseFloat(item.saldo) : 0;
                            const ada = item ? true : false;
                            
                            html += `
                                <div class="border rounded-lg p-4 ${ada ? 'bg-blue-50 border-blue-200' : 'bg-gray-50 border-gray-200'}">
                                    <div class="flex justify-between items-center">
                                        <span class="font-semibold text-gray-700">${bulanNama[i]}</span>
                                        <span class="text-lg font-bold ${ada ? 'text-blue-600' : 'text-gray-400'}">
                                            Rp ${saldo.toLocaleString('id-ID')}
                                        </span>
                                    </div>
                                </div>
                            `;
                        }
                        
                        html += '</div>';
                        
                        // Total
                        html += `
                            <div class="mt-6 bg-gradient-to-br from-purple-50 to-pink-50 border-2 border-purple-300 rounded-xl p-6">
                                <div class="flex justify-between items-center">
                                    <span class="text-lg font-bold text-gray-800">Total Saldo Tahunan:</span>
                                    <span class="text-3xl font-bold text-purple-600">Rp ${data.total.toLocaleString('id-ID')}</span>
                                </div>
                                <p class="text-sm text-gray-600 mt-2">
                                    <i class="fas fa-info-circle mr-1"></i>
                                    Dihitung dari ${data.jumlah_bulan} bulan yang memiliki data
                                </p>
                            </div>
                        `;
                    } else {
                        html += '<p class="text-center text-gray-500">Data tidak ditemukan</p>';
                    }
                    
                    html += '</div>';
                    document.getElementById('modalContent').innerHTML = html;
                    document.getElementById('modalDetail').classList.remove('hidden');
                })
                .catch(error => {
                    console.error('Error:', error);
                    alert('Terjadi kesalahan saat memuat data');
                });
        }

        function tutupModal() {
            document.getElementById('modalDetail').classList.add('hidden');
        }

        // Tutup modal jika klik di luar
        document.getElementById('modalDetail').addEventListener('click', function(e) {
            if (e.target === this) {
                tutupModal();
            }
        });
    </script>

</body>
</html>