<?php
session_start();
include 'config.php';

// Pastikan role adalah anggota
if (!isset($_SESSION['role']) || $_SESSION['role'] != 'Anggota') {
    header("Location: index.php");
    exit;
}

$id_anggota = $_SESSION['id_anggota'];

// Aktifkan error mysqli agar mudah debugging
mysqli_report(MYSQLI_REPORT_ERROR | MYSQLI_REPORT_STRICT);

// Query untuk mendapatkan data anggota
$anggotaQuery = "SELECT nama, nip, jabatan FROM anggota WHERE id_anggota = ?";
$stmt = $conn->prepare($anggotaQuery);
$stmt->bind_param("i", $id_anggota);
$stmt->execute();
$anggotaData = $stmt->get_result()->fetch_assoc();

// Filter tahun
$tahun_filter = isset($_GET['tahun']) ? $_GET['tahun'] : date('Y');

// Query untuk mendapatkan data SHU
$shuQuery = "SELECT * FROM shu_pembagian WHERE id_anggota = ?";
$params = [$id_anggota];
$types = "i";

if (!empty($tahun_filter)) {
    $shuQuery .= " AND tahun = ?";
    $params[] = $tahun_filter;
    $types .= "s";
}

$shuQuery .= " ORDER BY tahun DESC";

$stmt = $conn->prepare($shuQuery);
$stmt->bind_param($types, ...$params);
$stmt->execute();
$shuResult = $stmt->get_result();

// Hitung statistik
$totalSHUToko = 0;
$totalSHUUsipa = 0;
$totalSHUJasaUmum = 0;
$totalSHUKeseluruhan = 0;
$jumlahTahun = 0;

$shuData = [];
while ($row = $shuResult->fetch_assoc()) {
    $shuData[] = $row;
    $totalSHUToko += $row['shu_toko'];
    $totalSHUUsipa += $row['shu_usipa'];
    $totalSHUJasaUmum += $row['shu_jasa_umum'];
    $totalSHUKeseluruhan += $row['total_shu'];
    $jumlahTahun++;
}

// Query untuk mendapatkan daftar tahun yang tersedia
$tahunQuery = "SELECT DISTINCT tahun FROM shu_pembagian WHERE id_anggota = ? ORDER BY tahun DESC";
$stmtTahun = $conn->prepare($tahunQuery);
$stmtTahun->bind_param("i", $id_anggota);
$stmtTahun->execute();
$tahunResult = $stmtTahun->get_result();
$availableYears = [];
while ($row = $tahunResult->fetch_assoc()) {
    $availableYears[] = $row['tahun'];
}
?>

<!DOCTYPE html>
<html lang="id">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Pembagian SHU - Koperasi Pusdatin</title>
    <script src="https://cdn.tailwindcss.com"></script>
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.0.0/css/all.min.css">
    <script>
        tailwind.config = {
            theme: {
                extend: {
                    colors: {
                        primary: '#667eea',
                        secondary: '#764ba2',
                    }
                }
            }
        }
    </script>
    <style>
        @import url('https://fonts.googleapis.com/css2?family=Inter:wght@300;400;500;600;700;800&display=swap');
        body {
            font-family: 'Inter', sans-serif;
        }
    </style>
</head>
<body class="bg-gradient-to-br from-purple-50 to-blue-50">
    
    <!-- Sidebar -->
    <aside id="sidebar" class="fixed top-0 left-0 z-40 w-64 h-screen transition-transform -translate-x-full sm:translate-x-0">
        <div class="h-full px-3 py-4 overflow-y-auto bg-gradient-to-b from-purple-600 to-indigo-700 shadow-2xl">
            <!-- Logo Section -->
            <div class="flex flex-col items-center mb-8 pb-6 border-b border-white/20">
                <div class="w-24 h-24 bg-white rounded-full flex items-center justify-center mb-3 shadow-lg">
                    <img src="img/logo.png" alt="Logo Koperasi" class="w-20 h-20 object-contain" onerror="this.style.display='none'; this.nextElementSibling.style.display='block';">
                    <i class="fas fa-university text-purple-600 text-4xl hidden"></i>
                </div>
                <h2 class="text-white font-bold text-lg text-center">Koperasi Pusdatin</h2>
                <p class="text-purple-200 text-xs mt-1">Sistem Informasi Koperasi</p>
            </div>

            <!-- Profile Section -->
            <div class="mb-6 p-4 bg-white/10 rounded-lg backdrop-blur-sm">
                <div class="flex items-center space-x-3">
                    <div class="w-12 h-12 bg-white rounded-full flex items-center justify-center">
                        <i class="fas fa-user text-purple-600 text-xl"></i>
                    </div>
                    <div class="flex-1 min-w-0">
                        <p class="text-white font-semibold text-sm truncate"><?= htmlspecialchars($anggotaData['nama'] ?? 'Anggota') ?></p>
                        <p class="text-purple-200 text-xs truncate"><?= htmlspecialchars($anggotaData['nip'] ?? '-') ?></p>
                    </div>
                </div>
            </div>

            <!-- Navigation Menu -->
            <nav class="space-y-2">
                <a href="dashboard_anggota.php" class="flex items-center p-3 text-white rounded-lg hover:bg-white/20 transition-all duration-200 group">
                    <i class="fas fa-home w-6 text-center group-hover:scale-110 transition-transform"></i>
                    <span class="ml-3 font-medium">Dashboard</span>
                </a>
                
                <a href="ajukan_pinjaman.php" class="flex items-center p-3 text-white rounded-lg hover:bg-white/20 transition-all duration-200 group">
                    <i class="fas fa-money-bill-wave w-6 text-center group-hover:scale-110 transition-transform"></i>
                    <span class="ml-3 font-medium">Ajukan Pinjaman</span>
                </a>
                
                <a href="status_pinjaman.php" class="flex items-center p-3 text-white rounded-lg hover:bg-white/20 transition-all duration-200 group">
                    <i class="fas fa-file-contract w-6 text-center group-hover:scale-110 transition-transform"></i>
                    <span class="ml-3 font-medium">Status Pinjaman</span>
                </a>
                
                <a href="lihat_transaksi_defmart.php" class="flex items-center p-3 text-white rounded-lg hover:bg-white/20 transition-all duration-200 group">
                    <i class="fas fa-shopping-cart w-6 text-center group-hover:scale-110 transition-transform"></i>
                    <span class="ml-3 font-medium">Transaksi Defmart</span>
                </a>
                
                <a href="lihat_shu.php" class="flex items-center p-3 text-white bg-white/20 rounded-lg hover:bg-white/30 transition-all duration-200 group">
                    <i class="fas fa-chart-line w-6 text-center group-hover:scale-110 transition-transform"></i>
                    <span class="ml-3 font-medium">Pembagian SHU</span>
                </a>

                <div class="pt-4 mt-4 border-t border-white/20">
                    <a href="logout.php" class="flex items-center p-3 text-white rounded-lg hover:bg-red-500/30 transition-all duration-200 group">
                        <i class="fas fa-sign-out-alt w-6 text-center group-hover:scale-110 transition-transform"></i>
                        <span class="ml-3 font-medium">Logout</span>
                    </a>
                </div>
            </nav>
        </div>
    </aside>

    <!-- Main Content -->
    <div class="sm:ml-64">
        <!-- Top Navbar -->
        <nav class="bg-white shadow-md sticky top-0 z-30">
            <div class="px-4 py-3 lg:px-6">
                <div class="flex items-center justify-between">
                    <button id="toggleSidebar" class="inline-flex items-center p-2 text-sm text-gray-500 rounded-lg sm:hidden hover:bg-gray-100 focus:outline-none focus:ring-2 focus:ring-gray-200">
                        <i class="fas fa-bars text-xl"></i>
                    </button>
                    <h1 class="text-xl font-bold text-gray-800">Pembagian SHU</h1>
                    <div class="flex items-center space-x-4">
                        <span class="text-sm text-gray-600 hidden md:block">
                            <i class="far fa-calendar-alt mr-2"></i>
                            <?= date('d F Y') ?>
                        </span>
                    </div>
                </div>
            </div>
        </nav>

        <!-- Main Content Area -->
        <div class="p-4 lg:p-6">
            <!-- Info Card -->
            <div class="bg-gradient-to-r from-orange-500 to-yellow-500 rounded-2xl shadow-xl p-6 mb-6 text-white">
                <div class="flex items-center justify-between">
                    <div>
                        <h2 class="text-2xl font-bold mb-2">
                            <i class="fas fa-coins mr-2"></i>
                            Sisa Hasil Usaha (SHU)
                        </h2>
                        <p class="opacity-90 mb-1">
                            Bagian keuntungan tahunan yang dibagikan kepada anggota koperasi
                        </p>
                        <p class="opacity-90 text-sm">
                            <i class="fas fa-info-circle mr-2"></i>
                            SHU dihitung berdasarkan partisipasi anggota dalam usaha koperasi
                        </p>
                    </div>
                    <div class="hidden md:block">
                        <i class="fas fa-hand-holding-usd text-8xl opacity-20"></i>
                    </div>
                </div>
            </div>

            <!-- Filter Card -->
            <div class="bg-white rounded-xl shadow-lg p-6 mb-6">
                <h3 class="text-lg font-bold text-gray-800 flex items-center mb-4">
                    <i class="fas fa-filter text-purple-600 mr-2"></i>
                    Filter Tahun
                </h3>
                <form method="GET" class="flex gap-4 items-end">
                    <div class="flex-1">
                        <label class="block text-sm font-medium text-gray-700 mb-2">Pilih Tahun</label>
                        <select name="tahun" class="w-full px-4 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-purple-500 focus:border-transparent">
                            <option value="">Semua Tahun</option>
                            <?php
                            $tahun_sekarang = date('Y');
                            for ($i = $tahun_sekarang; $i >= ($tahun_sekarang - 10); $i--) {
                                $selected = ($tahun_filter == $i) ? 'selected' : '';
                                echo "<option value='$i' $selected>$i</option>";
                            }
                            ?>
                        </select>
                    </div>
                    <div class="flex gap-2">
                        <button type="submit" class="px-6 py-2 bg-purple-600 text-white rounded-lg hover:bg-purple-700 transition-colors font-medium">
                            <i class="fas fa-search mr-2"></i>Filter
                        </button>
                        <a href="?" class="px-6 py-2 bg-gray-200 text-gray-700 rounded-lg hover:bg-gray-300 transition-colors font-medium">
                            <i class="fas fa-redo mr-2"></i>Reset
                        </a>
                    </div>
                </form>
            </div>

            <!-- Statistics Cards -->
            <div class="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-4 gap-6 mb-6">
                <!-- SHU Toko -->
                <div class="bg-white rounded-xl shadow-lg hover:shadow-xl transition-all duration-300 overflow-hidden group">
                    <div class="p-6">
                        <div class="flex items-center justify-between mb-4">
                            <div class="p-3 bg-green-100 rounded-lg group-hover:scale-110 transition-transform">
                                <i class="fas fa-store text-green-600 text-2xl"></i>
                            </div>
                            <span class="text-xs font-semibold text-green-600 bg-green-100 px-3 py-1 rounded-full">TOKO</span>
                        </div>
                        <h3 class="text-gray-500 text-sm font-medium mb-2">SHU Toko</h3>
                        <p class="text-2xl font-bold text-gray-800 mb-1">Rp <?= number_format($totalSHUToko, 0, ',', '.') ?></p>
                        <p class="text-xs text-gray-400">
                            <i class="fas fa-shopping-bag mr-1"></i>
                            Dari transaksi toko
                        </p>
                    </div>
                    <div class="bg-green-50 px-6 py-3">
                        <p class="text-xs text-green-700">
                            <i class="fas fa-chart-line mr-1"></i>
                            Total akumulasi
                        </p>
                    </div>
                </div>

                <!-- SHU Usipa -->
                <div class="bg-white rounded-xl shadow-lg hover:shadow-xl transition-all duration-300 overflow-hidden group">
                    <div class="p-6">
                        <div class="flex items-center justify-between mb-4">
                            <div class="p-3 bg-blue-100 rounded-lg group-hover:scale-110 transition-transform">
                                <i class="fas fa-briefcase text-blue-600 text-2xl"></i>
                            </div>
                            <span class="text-xs font-semibold text-blue-600 bg-blue-100 px-3 py-1 rounded-full">USIPA</span>
                        </div>
                        <h3 class="text-gray-500 text-sm font-medium mb-2">SHU Usipa</h3>
                        <p class="text-2xl font-bold text-gray-800 mb-1">Rp <?= number_format($totalSHUUsipa, 0, ',', '.') ?></p>
                        <p class="text-xs text-gray-400">
                            <i class="fas fa-handshake mr-1"></i>
                            Dari usipa
                        </p>
                    </div>
                    <div class="bg-blue-50 px-6 py-3">
                        <p class="text-xs text-blue-700">
                            <i class="fas fa-chart-bar mr-1"></i>
                            Total akumulasi
                        </p>
                    </div>
                </div>

                <!-- SHU Jasa Umum -->
                <div class="bg-white rounded-xl shadow-lg hover:shadow-xl transition-all duration-300 overflow-hidden group">
                    <div class="p-6">
                        <div class="flex items-center justify-between mb-4">
                            <div class="p-3 bg-purple-100 rounded-lg group-hover:scale-110 transition-transform">
                                <i class="fas fa-cogs text-purple-600 text-2xl"></i>
                            </div>
                            <span class="text-xs font-semibold text-purple-600 bg-purple-100 px-3 py-1 rounded-full">JASA UMUM</span>
                        </div>
                        <h3 class="text-gray-500 text-sm font-medium mb-2">SHU Jasa Umum</h3>
                        <p class="text-2xl font-bold text-gray-800 mb-1">Rp <?= number_format($totalSHUJasaUmum, 0, ',', '.') ?></p>
                        <p class="text-xs text-gray-400">
                            <i class="fas fa-tools mr-1"></i>
                            Dari jasa umum
                        </p>
                    </div>
                    <div class="bg-purple-50 px-6 py-3">
                        <p class="text-xs text-purple-700">
                            <i class="fas fa-chart-area mr-1"></i>
                            Total akumulasi
                        </p>
                    </div>
                </div>

                <!-- Total SHU -->
                <div class="bg-white rounded-xl shadow-lg hover:shadow-xl transition-all duration-300 overflow-hidden group border-2 border-orange-200">
                    <div class="p-6">
                        <div class="flex items-center justify-between mb-4">
                            <div class="p-3 bg-orange-100 rounded-lg group-hover:scale-110 transition-transform">
                                <i class="fas fa-money-bill-wave text-orange-600 text-2xl"></i>
                            </div>
                            <span class="text-xs font-semibold text-orange-600 bg-orange-100 px-3 py-1 rounded-full">TOTAL</span>
                        </div>
                        <h3 class="text-gray-500 text-sm font-medium mb-2">Total SHU</h3>
                        <p class="text-2xl font-bold text-orange-600 mb-1">Rp <?= number_format($totalSHUKeseluruhan, 0, ',', '.') ?></p>
                        <p class="text-xs text-gray-400">
                            <i class="fas fa-calculator mr-1"></i>
                            Keseluruhan SHU
                        </p>
                    </div>
                    <div class="bg-orange-50 px-6 py-3">
                        <p class="text-xs text-orange-700">
                            <i class="fas fa-crown mr-1"></i>
                            Total yang diterima
                        </p>
                    </div>
                </div>
            </div>

            <!-- Data SHU Table -->
            <div class="bg-white rounded-xl shadow-lg overflow-hidden">
                <div class="p-6 border-b border-gray-200">
                    <h3 class="text-lg font-bold text-gray-800 flex items-center">
                        <i class="fas fa-table text-purple-600 mr-2"></i>
                        Riwayat Pembagian SHU
                    </h3>
                </div>

                <?php if (empty($shuData)): ?>
                    <div class="p-12 text-center">
                        <i class="fas fa-inbox text-gray-300 text-6xl mb-4"></i>
                        <p class="text-gray-500 text-lg font-medium">Belum Ada Data SHU</p>
                        <p class="text-gray-400 text-sm mt-2">Data pembagian SHU akan muncul di sini setelah pembagian dilakukan</p>
                    </div>
                <?php else: ?>
                    <div class="overflow-x-auto">
                        <table class="w-full">
                            <thead class="bg-gray-50">
                                <tr>
                                    <th class="px-6 py-4 text-left text-xs font-semibold text-gray-600 uppercase tracking-wider">No</th>
                                    <th class="px-6 py-4 text-left text-xs font-semibold text-gray-600 uppercase tracking-wider">Tahun</th>
                                    <th class="px-6 py-4 text-left text-xs font-semibold text-gray-600 uppercase tracking-wider">SHU Toko</th>
                                    <th class="px-6 py-4 text-left text-xs font-semibold text-gray-600 uppercase tracking-wider">SHU Usipa</th>
                                    <th class="px-6 py-4 text-left text-xs font-semibold text-gray-600 uppercase tracking-wider">SHU Jasa Umum</th>
                                    <th class="px-6 py-4 text-left text-xs font-semibold text-gray-600 uppercase tracking-wider">Total SHU</th>
                                    <th class="px-6 py-4 text-left text-xs font-semibold text-gray-600 uppercase tracking-wider">Aksi</th>
                                </tr>
                            </thead>
                            <tbody class="divide-y divide-gray-200">
                                <?php foreach ($shuData as $index => $shu): ?>
                                    <tr class="hover:bg-gray-50 transition-colors">
                                        <td class="px-6 py-4 whitespace-nowrap text-sm text-gray-800 font-medium"><?= $index + 1 ?></td>
                                        <td class="px-6 py-4 whitespace-nowrap">
                                            <span class="px-3 py-1 bg-purple-100 text-purple-700 rounded-full text-sm font-semibold">
                                                <i class="far fa-calendar mr-1"></i><?= $shu['tahun'] ?>
                                            </span>
                                        </td>
                                        <td class="px-6 py-4 whitespace-nowrap text-sm font-semibold text-green-600">
                                            Rp <?= number_format($shu['shu_toko'], 0, ',', '.') ?>
                                        </td>
                                        <td class="px-6 py-4 whitespace-nowrap text-sm font-semibold text-blue-600">
                                            Rp <?= number_format($shu['shu_usipa'], 0, ',', '.') ?>
                                        </td>
                                        <td class="px-6 py-4 whitespace-nowrap text-sm font-semibold text-purple-600">
                                            Rp <?= number_format($shu['shu_jasa_umum'], 0, ',', '.') ?>
                                        </td>
                                        <td class="px-6 py-4 whitespace-nowrap text-sm font-bold text-orange-600">
                                            Rp <?= number_format($shu['total_shu'], 0, ',', '.') ?>
                                        </td>
                                        <td class="px-6 py-4 whitespace-nowrap text-sm">
                                            <button onclick="showDetail(<?= htmlspecialchars(json_encode($shu)) ?>)" 
                                                    class="text-purple-600 hover:text-purple-900 font-medium">
                                                <i class="fas fa-eye mr-1"></i>
                                                Detail
                                            </button>
                                        </td>
                                    </tr>
                                <?php endforeach; ?>
                            </tbody>
                            <tfoot class="bg-gray-50">
                                <tr class="font-bold">
                                    <td colspan="2" class="px-6 py-4 text-sm text-gray-800 uppercase">Total Keseluruhan</td>
                                    <td class="px-6 py-4 text-sm text-green-600">Rp <?= number_format($totalSHUToko, 0, ',', '.') ?></td>
                                    <td class="px-6 py-4 text-sm text-blue-600">Rp <?= number_format($totalSHUUsipa, 0, ',', '.') ?></td>
                                    <td class="px-6 py-4 text-sm text-purple-600">Rp <?= number_format($totalSHUJasaUmum, 0, ',', '.') ?></td>
                                    <td class="px-6 py-4 text-sm text-orange-600">Rp <?= number_format($totalSHUKeseluruhan, 0, ',', '.') ?></td>
                                    <td></td>
                                </tr>
                            </tfoot>
                        </table>
                    </div>
                <?php endif; ?>
            </div>

            <!-- Info Section -->
            <div class="mt-6 bg-blue-50 border-l-4 border-blue-500 p-6 rounded-r-xl">
                <div class="flex">
                    <div class="flex-shrink-0">
                        <i class="fas fa-info-circle text-blue-500 text-2xl"></i>
                    </div>
                    <div class="ml-4">
                        <h4 class="text-blue-800 font-bold mb-2">Tentang SHU (Sisa Hasil Usaha)</h4>
                        <div class="text-blue-700 text-sm space-y-2">
                            <p><strong>SHU Toko:</strong> Bagian keuntungan dari transaksi di toko koperasi berdasarkan total pembelian anggota.</p>
                            <p><strong>SHU Usipa:</strong> Bagian keuntungan dari unit simpan pinjam berdasarkan partisipasi simpanan dan pinjaman.</p>
                            <p><strong>SHU Jasa Umum:</strong> Bagian keuntungan dari layanan umum koperasi lainnya.</p>
                            <p class="mt-3 pt-3 border-t border-blue-200"><i class="fas fa-lightbulb mr-2"></i><strong>Catatan:</strong> SHU dibagikan setahun sekali setelah RAT (Rapat Anggota Tahunan) dan dihitung berdasarkan partisipasi aktif anggota dalam kegiatan koperasi.</p>
                        </div>
                    </div>
                </div>
            </div>

            <!-- Footer -->
            <div class="mt-8 text-center text-gray-500 text-sm">
                <p>&copy; 2025 Koperasi Pusdatin. All rights reserved.</p>
            </div>
        </div>
    </div>

    <!-- Modal Detail -->
    <div id="detailModal" class="fixed inset-0 bg-gray-900 bg-opacity-50 hidden z-50 flex items-center justify-center p-4">
        <div class="bg-white rounded-2xl shadow-2xl max-w-2xl w-full max-h-[90vh] overflow-y-auto">
            <div class="p-6 border-b border-gray-200 bg-gradient-to-r from-orange-500 to-yellow-500">
                <div class="flex items-center justify-between">
                    <h3 class="text-xl font-bold text-white flex items-center">
                        <i class="fas fa-info-circle mr-2"></i>
                        Detail SHU
                    </h3>
                    <button onclick="closeModal()" class="text-white hover:text-gray-200 transition-colors">
                        <i class="fas fa-times text-2xl"></i>
                    </button>
                </div>
            </div>
            
            <div id="modalContent" class="p-6">
                <!-- Content will be inserted here -->
            </div>
        </div>
    </div>

    <script>
        // Toggle sidebar untuk mobile
        const sidebar = document.getElementById('sidebar');
        const toggleButton = document.getElementById('toggleSidebar');
        
        toggleButton.addEventListener('click', () => {
            sidebar.classList.toggle('-translate-x-full');
        });

        // Close sidebar when clicking outside on mobile
        document.addEventListener('click', (e) => {
            if (window.innerWidth < 640) {
                if (!sidebar.contains(e.target) && !toggleButton.contains(e.target)) {
                    sidebar.classList.add('-translate-x-full');
                }
            }
        });

        // Modal functions
        function showDetail(data) {
            const modal = document.getElementById('detailModal');
            const content = document.getElementById('modalContent');
            
            const persenToko = ((parseFloat(data.shu_toko) / parseFloat(data.total_shu)) * 100).toFixed(2);
            const persenUsipa = ((parseFloat(data.shu_usipa) / parseFloat(data.total_shu)) * 100).toFixed(2);
            const persenJasaUmum = ((parseFloat(data.shu_jasa_umum) / parseFloat(data.total_shu)) * 100).toFixed(2);
            
            content.innerHTML = `
                <div class="space-y-6">
                    <div class="flex items-center justify-center pb-4 border-b border-gray-200">
                        <div class="text-center">
                            <p class="text-sm text-gray-500 mb-1">Tahun</p>
                            <h4 class="text-3xl font-bold text-purple-600">${data.tahun}</h4>
                        </div>
                    </div>
                    
                    <div class="grid grid-cols-1 md:grid-cols-2 gap-4">
                        <div class="bg-green-50 p-4 rounded-lg border-2 border-green-200">
                            <div class="flex items-center justify-between mb-2">
                                <p class="text-xs text-gray-500 font-medium">SHU Toko</p>
                                <span class="text-xs font-bold text-green-600">${persenToko}%</span>
                            </div>
                            <p class="text-2xl font-bold text-green-600">
                                <i class="fas fa-store mr-2"></i>
                                Rp ${parseInt(data.shu_toko).toLocaleString('id-ID')}
                            </p>
                            <div class="mt-2 bg-green-200 rounded-full h-2">
                                <div class="bg-green-600 h-2 rounded-full" style="width: ${persenToko}%"></div>
                            </div>
                        </div>
                        
                        <div class="bg-blue-50 p-4 rounded-lg border-2 border-blue-200">
                            <div class="flex items-center justify-between mb-2">
                                <p class="text-xs text-gray-500 font-medium">SHU Usipa</p>
                                <span class="text-xs font-bold text-blue-600">${persenUsipa}%</span>
                            </div>
                            <p class="text-2xl font-bold text-blue-600">
                                <i class="fas fa-briefcase mr-2"></i>
                                Rp ${parseInt(data.shu_usipa).toLocaleString('id-ID')}
                            </p>
                            <div class="mt-2 bg-blue-200 rounded-full h-2">
                                <div class="bg-blue-600 h-2 rounded-full" style="width: ${persenUsipa}%"></div>
                            </div>
                        </div>
                        
                        <div class="bg-purple-50 p-4 rounded-lg border-2 border-purple-200">
                            <div class="flex items-center justify-between mb-2">
                                <p class="text-xs text-gray-500 font-medium">SHU Jasa Umum</p>
                                <span class="text-xs font-bold text-purple-600">${persenJasaUmum}%</span>
                            </div>
                            <p class="text-2xl font-bold text-purple-600">
                                <i class="fas fa-cogs mr-2"></i>
                                Rp ${parseInt(data.shu_jasa_umum).toLocaleString('id-ID')}
                            </p>
                            <div class="mt-2 bg-purple-200 rounded-full h-2">
                                <div class="bg-purple-600 h-2 rounded-full" style="width: ${persenJasaUmum}%"></div>
                            </div>
                        </div>
                        
                        <div class="bg-orange-50 p-4 rounded-lg border-2 border-orange-200">
                            <p class="text-xs text-gray-500 mb-2 font-medium">Total SHU</p>
                            <p class="text-2xl font-bold text-orange-600">
                                <i class="fas fa-money-bill-wave mr-2"></i>
                                Rp ${parseInt(data.total_shu).toLocaleString('id-ID')}
                            </p>
                            <p class="text-xs text-orange-700 mt-2">
                                <i class="fas fa-check-circle mr-1"></i>
                                Total yang diterima
                            </p>
                        </div>
                    </div>
                    
                    <div class="bg-gradient-to-r from-orange-100 to-yellow-100 p-4 rounded-lg">
                        <h5 class="font-bold text-gray-800 mb-3 flex items-center">
                            <i class="fas fa-chart-pie text-orange-600 mr-2"></i>
                            Komposisi SHU
                        </h5>
                        <div class="space-y-2">
                            <div class="flex justify-between items-center">
                                <span class="text-sm text-gray-700">
                                    <i class="fas fa-circle text-green-600 text-xs mr-2"></i>
                                    SHU Toko
                                </span>
                                <span class="text-sm font-bold text-gray-800">${persenToko}%</span>
                            </div>
                            <div class="flex justify-between items-center">
                                <span class="text-sm text-gray-700">
                                    <i class="fas fa-circle text-blue-600 text-xs mr-2"></i>
                                    SHU Usipa
                                </span>
                                <span class="text-sm font-bold text-gray-800">${persenUsipa}%</span>
                            </div>
                            <div class="flex justify-between items-center">
                                <span class="text-sm text-gray-700">
                                    <i class="fas fa-circle text-purple-600 text-xs mr-2"></i>
                                    SHU Jasa Umum
                                </span>
                                <span class="text-sm font-bold text-gray-800">${persenJasaUmum}%</span>
                            </div>
                        </div>
                    </div>
                    
                    <div class="bg-blue-50 p-4 rounded-lg border-l-4 border-blue-500">
                        <p class="text-sm text-blue-800">
                            <i class="fas fa-info-circle mr-2"></i>
                            <strong>Informasi:</strong> SHU tahun ${data.tahun} telah dibagikan sesuai dengan partisipasi Anda dalam kegiatan koperasi selama tahun tersebut.
                        </p>
                    </div>
                </div>
            `;
            
            modal.classList.remove('hidden');
        }
        
        function closeModal() {
            const modal = document.getElementById('detailModal');
            modal.classList.add('hidden');
        }
        
        // Close modal when clicking outside
        document.getElementById('detailModal').addEventListener('click', function(e) {
            if (e.target === this) {
                closeModal();
            }
        });
        
        // Close modal with Escape key
        document.addEventListener('keydown', function(e) {
            if (e.key === 'Escape') {
                closeModal();
            }
        });
    </script>
</body>
</html>