<?php
session_start();
include 'config.php';

// Pastikan user login dan rolenya Anggota
if (!isset($_SESSION['role']) || $_SESSION['role'] != 'Anggota') {
    header("Location: index.php");
    exit;
}

// Ambil ID anggota yang sedang login
$id_anggota = $_SESSION['id_anggota'];

// Aktifkan error mysqli agar mudah debugging
mysqli_report(MYSQLI_REPORT_ERROR | MYSQLI_REPORT_STRICT);

// Query untuk mendapatkan data anggota
$anggotaQuery = "SELECT nama, nip, jabatan FROM anggota WHERE id_anggota = ?";
$stmt = $conn->prepare($anggotaQuery);
$stmt->bind_param("i", $id_anggota);
$stmt->execute();
$anggotaData = $stmt->get_result()->fetch_assoc();

// Filter bulan dan tahun
$bulan_filter = isset($_GET['bulan']) ? $_GET['bulan'] : date('m');
$tahun_filter = isset($_GET['tahun']) ? $_GET['tahun'] : date('Y');

// Query untuk mengambil data transaksi dengan filter
$query = "SELECT * FROM defmart_transaksi WHERE id_anggota = ?";
$params = [$id_anggota];
$types = "i";

if (!empty($bulan_filter) && !empty($tahun_filter)) {
    $query .= " AND MONTH(tanggal_transaksi) = ? AND YEAR(tanggal_transaksi) = ?";
    $params[] = $bulan_filter;
    $params[] = $tahun_filter;
    $types .= "ii";
}

$query .= " ORDER BY tanggal_transaksi DESC";

$stmt = $conn->prepare($query);
$stmt->bind_param($types, ...$params);
$stmt->execute();
$result = $stmt->get_result();

// Hitung statistik
$stats_query = "SELECT 
    COUNT(*) as total_transaksi,
    SUM(total_harga) as total_belanja,
    AVG(total_harga) as rata_rata,
    SUM(qty) as total_item
    FROM defmart_transaksi 
    WHERE id_anggota = ?";

$stats_params = [$id_anggota];
$stats_types = "i";

if (!empty($bulan_filter) && !empty($tahun_filter)) {
    $stats_query .= " AND MONTH(tanggal_transaksi) = ? AND YEAR(tanggal_transaksi) = ?";
    $stats_params[] = $bulan_filter;
    $stats_params[] = $tahun_filter;
    $stats_types .= "ii";
}

$stats_stmt = $conn->prepare($stats_query);
$stats_stmt->bind_param($stats_types, ...$stats_params);
$stats_stmt->execute();
$stats_result = $stats_stmt->get_result();
$stats = $stats_result->fetch_assoc();

// Query untuk mendapatkan top 3 barang terlaris
$top_items_query = "SELECT nama_barang, SUM(qty) as total_qty, COUNT(*) as jumlah_transaksi 
                    FROM defmart_transaksi 
                    WHERE id_anggota = ?";
$top_params = [$id_anggota];
$top_types = "i";

if (!empty($bulan_filter) && !empty($tahun_filter)) {
    $top_items_query .= " AND MONTH(tanggal_transaksi) = ? AND YEAR(tanggal_transaksi) = ?";
    $top_params[] = $bulan_filter;
    $top_params[] = $tahun_filter;
    $top_types .= "ii";
}

$top_items_query .= " GROUP BY nama_barang ORDER BY total_qty DESC LIMIT 3";

$top_stmt = $conn->prepare($top_items_query);
$top_stmt->bind_param($top_types, ...$top_params);
$top_stmt->execute();
$top_items = $top_stmt->get_result();
?>

<!DOCTYPE html>
<html lang="id">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Transaksi Defmart - Koperasi Pusdatin</title>
    <script src="https://cdn.tailwindcss.com"></script>
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.0.0/css/all.min.css">
    <script>
        tailwind.config = {
            theme: {
                extend: {
                    colors: {
                        primary: '#667eea',
                        secondary: '#764ba2',
                    }
                }
            }
        }
    </script>
    <style>
        @import url('https://fonts.googleapis.com/css2?family=Inter:wght@300;400;500;600;700;800&display=swap');
        body {
            font-family: 'Inter', sans-serif;
        }
    </style>
</head>
<body class="bg-gradient-to-br from-purple-50 to-blue-50">
    
    <!-- Sidebar -->
    <aside id="sidebar" class="fixed top-0 left-0 z-40 w-64 h-screen transition-transform -translate-x-full sm:translate-x-0">
        <div class="h-full px-3 py-4 overflow-y-auto bg-gradient-to-b from-purple-600 to-indigo-700 shadow-2xl">
            <!-- Logo Section -->
            <div class="flex flex-col items-center mb-8 pb-6 border-b border-white/20">
                <div class="w-24 h-24 bg-white rounded-full flex items-center justify-center mb-3 shadow-lg">
                    <img src="img/logo.png" alt="Logo Koperasi" class="w-20 h-20 object-contain" onerror="this.style.display='none'; this.nextElementSibling.style.display='block';">
                    <i class="fas fa-university text-purple-600 text-4xl hidden"></i>
                </div>
                <h2 class="text-white font-bold text-lg text-center">Koperasi Pusdatin</h2>
                <p class="text-purple-200 text-xs mt-1">Sistem Informasi Koperasi</p>
            </div>

            <!-- Profile Section -->
            <div class="mb-6 p-4 bg-white/10 rounded-lg backdrop-blur-sm">
                <div class="flex items-center space-x-3">
                    <div class="w-12 h-12 bg-white rounded-full flex items-center justify-center">
                        <i class="fas fa-user text-purple-600 text-xl"></i>
                    </div>
                    <div class="flex-1 min-w-0">
                        <p class="text-white font-semibold text-sm truncate"><?= htmlspecialchars($anggotaData['nama'] ?? 'Anggota') ?></p>
                        <p class="text-purple-200 text-xs truncate"><?= htmlspecialchars($anggotaData['nip'] ?? '-') ?></p>
                    </div>
                </div>
            </div>

            <!-- Navigation Menu -->
            <nav class="space-y-2">
                <a href="dashboard_anggota.php" class="flex items-center p-3 text-white rounded-lg hover:bg-white/20 transition-all duration-200 group">
                    <i class="fas fa-home w-6 text-center group-hover:scale-110 transition-transform"></i>
                    <span class="ml-3 font-medium">Dashboard</span>
                </a>
                
                <a href="ajukan_pinjaman.php" class="flex items-center p-3 text-white rounded-lg hover:bg-white/20 transition-all duration-200 group">
                    <i class="fas fa-money-bill-wave w-6 text-center group-hover:scale-110 transition-transform"></i>
                    <span class="ml-3 font-medium">Ajukan Pinjaman</span>
                </a>
                
                <a href="status_pinjaman.php" class="flex items-center p-3 text-white rounded-lg hover:bg-white/20 transition-all duration-200 group">
                    <i class="fas fa-file-contract w-6 text-center group-hover:scale-110 transition-transform"></i>
                    <span class="ml-3 font-medium">Status Pinjaman</span>
                </a>
                
                <a href="lihat_transaksi_defmart.php" class="flex items-center p-3 text-white bg-white/20 rounded-lg hover:bg-white/30 transition-all duration-200 group">
                    <i class="fas fa-shopping-cart w-6 text-center group-hover:scale-110 transition-transform"></i>
                    <span class="ml-3 font-medium">Transaksi Defmart</span>
                </a>
                
                <a href="lihat_shu.php" class="flex items-center p-3 text-white rounded-lg hover:bg-white/20 transition-all duration-200 group">
                    <i class="fas fa-chart-line w-6 text-center group-hover:scale-110 transition-transform"></i>
                    <span class="ml-3 font-medium">Pembagian SHU</span>
                </a>

                <div class="pt-4 mt-4 border-t border-white/20">
                    <a href="logout.php" class="flex items-center p-3 text-white rounded-lg hover:bg-red-500/30 transition-all duration-200 group">
                        <i class="fas fa-sign-out-alt w-6 text-center group-hover:scale-110 transition-transform"></i>
                        <span class="ml-3 font-medium">Logout</span>
                    </a>
                </div>
            </nav>
        </div>
    </aside>

    <!-- Main Content -->
    <div class="sm:ml-64">
        <!-- Top Navbar -->
        <nav class="bg-white shadow-md sticky top-0 z-30">
            <div class="px-4 py-3 lg:px-6">
                <div class="flex items-center justify-between">
                    <button id="toggleSidebar" class="inline-flex items-center p-2 text-sm text-gray-500 rounded-lg sm:hidden hover:bg-gray-100 focus:outline-none focus:ring-2 focus:ring-gray-200">
                        <i class="fas fa-bars text-xl"></i>
                    </button>
                    <h1 class="text-xl font-bold text-gray-800">Riwayat Transaksi Defmart</h1>
                    <div class="flex items-center space-x-4">
                        <span class="text-sm text-gray-600 hidden md:block">
                            <i class="far fa-calendar-alt mr-2"></i>
                            <?= date('d F Y') ?>
                        </span>
                    </div>
                </div>
            </div>
        </nav>

        <!-- Main Content Area -->
        <div class="p-4 lg:p-6">
            <!-- Filter Card -->
            <div class="bg-white rounded-xl shadow-lg p-6 mb-6">
                <div class="flex items-center justify-between mb-4">
                    <h3 class="text-lg font-bold text-gray-800 flex items-center">
                        <i class="fas fa-filter text-purple-600 mr-2"></i>
                        Filter Transaksi
                    </h3>
                    <button class="px-4 py-2 bg-green-600 text-white rounded-lg hover:bg-green-700 transition-colors text-sm font-medium" onclick="exportToExcel()">
                        <i class="fas fa-file-excel mr-2"></i>
                        Export to XLSX
                    </button>
                </div>
                <form method="GET" class="grid grid-cols-1 md:grid-cols-3 gap-4">
                    <div>
                        <label class="block text-sm font-medium text-gray-700 mb-2">Bulan</label>
                        <select name="bulan" class="w-full px-4 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-purple-500 focus:border-transparent">
                            <option value="">Semua Bulan</option>
                            <?php
                            $bulan_nama = [
                                1 => 'Januari', 2 => 'Februari', 3 => 'Maret', 4 => 'April',
                                5 => 'Mei', 6 => 'Juni', 7 => 'Juli', 8 => 'Agustus',
                                9 => 'September', 10 => 'Oktober', 11 => 'November', 12 => 'Desember'
                            ];
                            for ($i = 1; $i <= 12; $i++) {
                                $selected = ($bulan_filter == $i) ? 'selected' : '';
                                echo "<option value='$i' $selected>{$bulan_nama[$i]}</option>";
                            }
                            ?>
                        </select>
                    </div>
                    <div>
                        <label class="block text-sm font-medium text-gray-700 mb-2">Tahun</label>
                        <select name="tahun" class="w-full px-4 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-purple-500 focus:border-transparent">
                            <option value="">Semua Tahun</option>
                            <?php
                            $tahun_sekarang = date('Y');
                            for ($i = $tahun_sekarang; $i >= ($tahun_sekarang - 5); $i--) {
                                $selected = ($tahun_filter == $i) ? 'selected' : '';
                                echo "<option value='$i' $selected>$i</option>";
                            }
                            ?>
                        </select>
                    </div>
                    <div class="flex items-end gap-2">
                        <button type="submit" class="flex-1 px-4 py-2 bg-purple-600 text-white rounded-lg hover:bg-purple-700 transition-colors font-medium">
                            <i class="fas fa-search mr-2"></i>Filter
                        </button>
                        <a href="?" class="px-4 py-2 bg-gray-200 text-gray-700 rounded-lg hover:bg-gray-300 transition-colors font-medium">
                            <i class="fas fa-redo mr-2"></i>Reset
                        </a>
                    </div>
                </form>
            </div>

            <!-- Statistics Cards -->
            <div class="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-4 gap-6 mb-6">
                <!-- Total Transaksi -->
                <div class="bg-white rounded-xl shadow-lg hover:shadow-xl transition-all duration-300 overflow-hidden group">
                    <div class="p-6">
                        <div class="flex items-center justify-between mb-4">
                            <div class="p-3 bg-blue-100 rounded-lg group-hover:scale-110 transition-transform">
                                <i class="fas fa-shopping-bag text-blue-600 text-2xl"></i>
                            </div>
                        </div>
                        <h3 class="text-gray-500 text-sm font-medium mb-2">Total Transaksi</h3>
                        <p class="text-3xl font-bold text-gray-800"><?= number_format($stats['total_transaksi']) ?></p>
                    </div>
                    <div class="bg-blue-50 px-6 py-3">
                        <p class="text-xs text-blue-700">
                            <i class="fas fa-receipt mr-1"></i>
                            Total pembelian
                        </p>
                    </div>
                </div>

                <!-- Total Belanja -->
                <div class="bg-white rounded-xl shadow-lg hover:shadow-xl transition-all duration-300 overflow-hidden group">
                    <div class="p-6">
                        <div class="flex items-center justify-between mb-4">
                            <div class="p-3 bg-green-100 rounded-lg group-hover:scale-110 transition-transform">
                                <i class="fas fa-money-bill-wave text-green-600 text-2xl"></i>
                            </div>
                        </div>
                        <h3 class="text-gray-500 text-sm font-medium mb-2">Total Belanja</h3>
                        <p class="text-2xl font-bold text-gray-800">Rp <?= number_format($stats['total_belanja'] ?: 0, 0, ',', '.') ?></p>
                    </div>
                    <div class="bg-green-50 px-6 py-3">
                        <p class="text-xs text-green-700">
                            <i class="fas fa-wallet mr-1"></i>
                            Total pengeluaran
                        </p>
                    </div>
                </div>

                <!-- Rata-rata -->
                <div class="bg-white rounded-xl shadow-lg hover:shadow-xl transition-all duration-300 overflow-hidden group">
                    <div class="p-6">
                        <div class="flex items-center justify-between mb-4">
                            <div class="p-3 bg-yellow-100 rounded-lg group-hover:scale-110 transition-transform">
                                <i class="fas fa-chart-bar text-yellow-600 text-2xl"></i>
                            </div>
                        </div>
                        <h3 class="text-gray-500 text-sm font-medium mb-2">Rata-rata</h3>
                        <p class="text-2xl font-bold text-gray-800">Rp <?= number_format($stats['rata_rata'] ?: 0, 0, ',', '.') ?></p>
                    </div>
                    <div class="bg-yellow-50 px-6 py-3">
                        <p class="text-xs text-yellow-700">
                            <i class="fas fa-calculator mr-1"></i>
                            Per transaksi
                        </p>
                    </div>
                </div>

                <!-- Total Item -->
                <div class="bg-white rounded-xl shadow-lg hover:shadow-xl transition-all duration-300 overflow-hidden group">
                    <div class="p-6">
                        <div class="flex items-center justify-between mb-4">
                            <div class="p-3 bg-purple-100 rounded-lg group-hover:scale-110 transition-transform">
                                <i class="fas fa-boxes text-purple-600 text-2xl"></i>
                            </div>
                        </div>
                        <h3 class="text-gray-500 text-sm font-medium mb-2">Total Item</h3>
                        <p class="text-3xl font-bold text-gray-800"><?= number_format($stats['total_item'] ?: 0) ?></p>
                    </div>
                    <div class="bg-purple-50 px-6 py-3">
                        <p class="text-xs text-purple-700">
                            <i class="fas fa-box mr-1"></i>
                            Item dibeli
                        </p>
                    </div>
                </div>
            </div>

            <div class="grid grid-cols-1 lg:grid-cols-3 gap-6">
                <!-- Transaction Table -->
                <div class="lg:col-span-2 bg-white rounded-xl shadow-lg overflow-hidden">
                    <div class="p-6 border-b border-gray-200">
                        <h3 class="text-lg font-bold text-gray-800 flex items-center">
                            <i class="fas fa-list-alt text-purple-600 mr-2"></i>
                            Daftar Transaksi
                        </h3>
                    </div>

                    <?php if ($result->num_rows > 0): ?>
                        <div class="overflow-x-auto">
                            <table class="w-full" id="transactionTable">
                                <thead class="bg-gray-50">
                                    <tr>
                                        <th class="px-6 py-4 text-left text-xs font-semibold text-gray-600 uppercase tracking-wider">No</th>
                                        <th class="px-6 py-4 text-left text-xs font-semibold text-gray-600 uppercase tracking-wider">Nama Barang</th>
                                        <th class="px-6 py-4 text-left text-xs font-semibold text-gray-600 uppercase tracking-wider">QTY</th>
                                        <th class="px-6 py-4 text-left text-xs font-semibold text-gray-600 uppercase tracking-wider">Harga Satuan</th>
                                        <th class="px-6 py-4 text-left text-xs font-semibold text-gray-600 uppercase tracking-wider">Total</th>
                                        <th class="px-6 py-4 text-left text-xs font-semibold text-gray-600 uppercase tracking-wider">Pembayaran</th>
                                        <th class="px-6 py-4 text-left text-xs font-semibold text-gray-600 uppercase tracking-wider">Tanggal</th>
                                    </tr>
                                </thead>
                                <tbody class="divide-y divide-gray-200">
                                    <?php
                                    $no = 1;
                                    $result->data_seek(0);
                                    while ($row = $result->fetch_assoc()) {
                                        echo "<tr class='hover:bg-gray-50 transition-colors'>
                                            <td class='px-6 py-4 whitespace-nowrap text-sm text-gray-800 font-medium'>$no</td>
                                            <td class='px-6 py-4 text-sm font-semibold text-gray-800'>" . htmlspecialchars($row['nama_barang']) . "</td>
                                            <td class='px-6 py-4 whitespace-nowrap'><span class='px-2 py-1 bg-purple-100 text-purple-700 rounded-full text-xs font-semibold'>{$row['qty']}</span></td>
                                            <td class='px-6 py-4 whitespace-nowrap text-sm text-gray-600'>Rp " . number_format($row['harga_satuan'], 0, ',', '.') . "</td>
                                            <td class='px-6 py-4 whitespace-nowrap text-sm font-bold text-green-600'>Rp " . number_format($row['total_harga'], 0, ',', '.') . "</td>
                                            <td class='px-6 py-4 whitespace-nowrap'><span class='px-3 py-1 bg-blue-100 text-blue-700 rounded-full text-xs font-semibold'>" . htmlspecialchars($row['metode_pembayaran']) . "</span></td>
                                            <td class='px-6 py-4 whitespace-nowrap text-sm text-gray-600'>" . date('d/m/Y H:i', strtotime($row['tanggal_transaksi'])) . "</td>
                                        </tr>";
                                        $no++;
                                    }
                                    ?>
                                </tbody>
                            </table>
                        </div>
                    <?php else: ?>
                        <div class="p-12 text-center">
                            <i class="fas fa-shopping-cart text-gray-300 text-6xl mb-4"></i>
                            <p class="text-gray-500 text-lg font-medium">Belum Ada Transaksi</p>
                            <p class="text-gray-400 text-sm mt-2">Transaksi Anda akan muncul di sini setelah melakukan pembelian</p>
                        </div>
                    <?php endif; ?>
                </div>

                <!-- Sidebar -->
                <div class="space-y-6">
                    <!-- Top Items Card -->
                    <div class="bg-white rounded-xl shadow-lg overflow-hidden">
                        <div class="p-6 border-b border-gray-200 bg-gradient-to-r from-orange-500 to-yellow-500">
                            <h3 class="text-lg font-bold text-white flex items-center">
                                <i class="fas fa-trophy mr-2"></i>
                                Top 3 Barang Favorit
                            </h3>
                        </div>
                        <div class="p-6">
                            <?php if ($top_items->num_rows > 0): ?>
                                <div class="space-y-3">
                                    <?php 
                                    $rank = 1;
                                    $medals = ['🥇', '🥈', '🥉'];
                                    $colors = ['from-yellow-100 to-orange-100', 'from-gray-100 to-gray-200', 'from-orange-100 to-yellow-100'];
                                    while ($item = $top_items->fetch_assoc()): 
                                    ?>
                                        <div class="bg-gradient-to-r <?= $colors[$rank-1] ?> p-4 rounded-lg">
                                            <div class="flex items-center">
                                                <span class="text-2xl mr-3"><?= $medals[$rank-1] ?></span>
                                                <div class="flex-1">
                                                    <h6 class="font-bold text-gray-800 mb-1"><?= htmlspecialchars($item['nama_barang']) ?></h6>
                                                    <p class="text-xs text-gray-600">
                                                        <i class="fas fa-box mr-1"></i><?= $item['total_qty'] ?> item • 
                                                        <i class="fas fa-receipt ml-2 mr-1"></i><?= $item['jumlah_transaksi'] ?> transaksi
                                                    </p>
                                                </div>
                                            </div>
                                        </div>
                                    <?php 
                                    $rank++;
                                    endwhile; 
                                    ?>
                                </div>
                            <?php else: ?>
                                <div class="text-center text-gray-400 py-8">
                                    <i class="fas fa-medal text-5xl mb-3 opacity-30"></i>
                                    <p class="text-sm">Belum ada data favorit</p>
                                </div>
                            <?php endif; ?>
                        </div>
                    </div>

                    <!-- Quick Actions Card -->
                    <div class="bg-white rounded-xl shadow-lg overflow-hidden">
                        <div class="p-6 border-b border-gray-200">
                            <h3 class="text-lg font-bold text-gray-800 flex items-center">
                                <i class="fas fa-bolt text-yellow-500 mr-2"></i>
                                Aksi Cepat
                            </h3>
                        </div>
                        <div class="p-6">
                            <a href="defmart.php" class="block w-full px-4 py-3 bg-gradient-to-r from-green-500 to-emerald-600 text-white rounded-lg hover:from-green-600 hover:to-emerald-700 transition-all duration-300 text-center font-medium shadow-lg hover:shadow-xl">
                                <i class="fas fa-shopping-cart mr-2"></i>
                                Belanja Lagi
                            </a>
                        </div>
                    </div>
                </div>
            </div>

            <!-- Footer -->
            <div class="mt-8 text-center text-gray-500 text-sm">
                <p>&copy; 2025 Koperasi Pusdatin. All rights reserved.</p>
            </div>
        </div>
    </div>

    <script>
        // Toggle sidebar untuk mobile
        const sidebar = document.getElementById('sidebar');
        const toggleButton = document.getElementById('toggleSidebar');
        
        toggleButton.addEventListener('click', () => {
            sidebar.classList.toggle('-translate-x-full');
        });

        // Close sidebar when clicking outside on mobile
        document.addEventListener('click', (e) => {
            if (window.innerWidth < 640) {
                if (!sidebar.contains(e.target) && !toggleButton.contains(e.target)) {
                    sidebar.classList.add('-translate-x-full');
                }
            }
        });

        function exportToExcel() {
            const form = document.createElement('form');
            form.method = 'POST';
            form.action = 'export_riwayat_anggota.php';
            form.target = '_blank';
            
            const bulan = '<?= $bulan_filter ?>';
            const tahun = '<?= $tahun_filter ?>';
            
            if (bulan) {
                const inputBulan = document.createElement('input');
                inputBulan.type = 'hidden';
                inputBulan.name = 'bulan';
                inputBulan.value = bulan;
                form.appendChild(inputBulan);
            }
            
            if (tahun) {
                const inputTahun = document.createElement('input');
                inputTahun.type = 'hidden';
                inputTahun.name = 'tahun';
                inputTahun.value = tahun;
                form.appendChild(inputTahun);
            }
            
            document.body.appendChild(form);
            form.submit();
            document.body.removeChild(form);
        }
    </script>
</body>
</html>