<?php
session_start();
include 'config.php';

error_reporting(E_ALL);
ini_set('display_errors', 1);

// Cek apakah user login dan role = Kepala Unit (id_role = 7)
if (!isset($_SESSION['role']) || $_SESSION['id_role'] != 7) {
    header("Location: index.php");
    exit;
}

$id_unit = 1; 
$nama_unit = "Simpan Pinjam";

// Array nama bulan
$bulan_nama = ['', 'Januari', 'Februari', 'Maret', 'April', 'Mei', 'Juni', 
              'Juli', 'Agustus', 'September', 'Oktober', 'November', 'Desember'];

// Proses pemotongan gaji otomatis
if (isset($_POST['potong_gaji_otomatis'])) {
    $tahun = intval($_POST['tahun_potong']);
    $bulan = intval($_POST['bulan_potong']);
    $id_user = $_SESSION['id_user'];
    
    // Validasi: Cek apakah bulan yang dipilih sudah lewat atau bulan ini
    $bulan_sekarang = intval(date('n'));
    $tahun_sekarang = intval(date('Y'));
    
    // Jika tahun yang dipilih lebih besar dari tahun sekarang, tolak
    if ($tahun > $tahun_sekarang) {
        header("Location: potong_gaji.php?error=bulan_belum_tiba");
        exit;
    }
    
    // Jika tahun sama, cek bulan
    if ($tahun == $tahun_sekarang && $bulan > $bulan_sekarang) {
        header("Location: potong_gaji.php?error=bulan_belum_tiba");
        exit;
    }
    
    // Cek apakah sudah pernah potong gaji di bulan dan tahun yang sama
    $stmt_cek = $conn->prepare("SELECT COUNT(*) as total FROM simpanan 
        WHERE YEAR(tanggal_setor) = ? AND MONTH(tanggal_setor) = ? 
        AND jenis = 'wajib' 
        AND (keterangan LIKE '%Pemotongan Gaji Otomatis%' OR keterangan LIKE '%Potong Gaji%')");
    
    if (!$stmt_cek) {
        header("Location: potong_gaji.php?error=gagal&msg=" . urlencode("Prepare statement gagal: " . $conn->error));
        exit;
    }
    
    $stmt_cek->bind_param("ii", $tahun, $bulan);
    
    if (!$stmt_cek->execute()) {
        header("Location: potong_gaji.php?error=gagal&msg=" . urlencode("Execute statement gagal: " . $stmt_cek->error));
        exit;
    }
    
    $result_cek = $stmt_cek->get_result();
    
    if ($result_cek) {
        $data_cek = $result_cek->fetch_assoc();
        if ($data_cek['total'] > 0) {
            $stmt_cek->close();
            header("Location: potong_gaji.php?error=sudah_potong");
            exit;
        }
    }
    $stmt_cek->close();
    
    // Ambil semua anggota yang memiliki gaji pokok
    $stmt_anggota = $conn->prepare("SELECT id_anggota, nama, gaji FROM anggota WHERE gaji > 0 AND gaji IS NOT NULL");
    
    if (!$stmt_anggota) {
        header("Location: potong_gaji.php?error=gagal&msg=" . urlencode("Prepare statement anggota gagal: " . $conn->error));
        exit;
    }
    
    if (!$stmt_anggota->execute()) {
        header("Location: potong_gaji.php?error=gagal&msg=" . urlencode("Execute statement anggota gagal: " . $stmt_anggota->error));
        exit;
    }
    
    $result_anggota = $stmt_anggota->get_result();
    
    if ($result_anggota && $result_anggota->num_rows > 0) {
        $conn->begin_transaction();
        
        try {
            $total_dipotong = 0;
            $jumlah_anggota = 0;
            $tanggal_setor = sprintf("%d-%02d-01", $tahun, $bulan);
            
            // Nominal simpanan wajib tetap Rp 50.000
            $nominal_simpanan = 50000;
            
            // Prepared statements untuk insert simpanan
            $stmt_simpanan = $conn->prepare("INSERT INTO simpanan (id_anggota, jenis, nominal, tanggal_setor, keterangan) 
                VALUES (?, 'wajib', ?, ?, ?)");
            
            if (!$stmt_simpanan) {
                throw new Exception("Prepare simpanan gagal: " . $conn->error);
            }
            
            // ========== PREPARED STATEMENTS UNTUK TABUNGAN_USIPA ==========
            
            // Cek apakah sudah ada record tabungan_usipa untuk anggota, tahun, dan bulan tertentu
            $stmt_cek_tabungan = $conn->prepare("SELECT id_tabungan, setoran, saldo FROM tabungan_usipa 
                WHERE id_anggota = ? AND tahun = ? AND bulan = ?");
            
            if (!$stmt_cek_tabungan) {
                throw new Exception("Prepare cek tabungan_usipa gagal: " . $conn->error);
            }
            
            // Update tabungan_usipa yang sudah ada
            $stmt_update_tabungan = $conn->prepare("UPDATE tabungan_usipa SET 
                setoran = ?, 
                saldo = ?,
                tanggal_transaksi = ?
                WHERE id_tabungan = ?");
            
            if (!$stmt_update_tabungan) {
                throw new Exception("Prepare update tabungan_usipa gagal: " . $conn->error);
            }
            
            // Insert tabungan_usipa baru
            $stmt_insert_tabungan = $conn->prepare("INSERT INTO tabungan_usipa 
                (id_anggota, tahun, bulan, setoran, pengambilan, saldo, tanggal_transaksi) 
                VALUES (?, ?, ?, ?, 0.00, ?, ?)");
            
            if (!$stmt_insert_tabungan) {
                throw new Exception("Prepare insert tabungan_usipa gagal: " . $conn->error);
            }
            
            // ========== AKHIR PREPARED STATEMENTS TABUNGAN_USIPA ==========
            
            while ($anggota = $result_anggota->fetch_assoc()) {
                // Insert ke tabel simpanan
                $keterangan = "Pemotongan Gaji Otomatis - " . $bulan_nama[$bulan] . " " . $tahun;
                $stmt_simpanan->bind_param("idss", $anggota['id_anggota'], $nominal_simpanan, $tanggal_setor, $keterangan);
                
                if (!$stmt_simpanan->execute()) {
                    throw new Exception("Error inserting simpanan untuk anggota ID " . $anggota['id_anggota'] . ": " . $stmt_simpanan->error);
                }
                
                // ========== INSERT/UPDATE KE TABEL TABUNGAN_USIPA ==========
                
                // Cek apakah sudah ada record untuk anggota ini di bulan dan tahun yang sama
                $stmt_cek_tabungan->bind_param("iii", $anggota['id_anggota'], $tahun, $bulan);
                $stmt_cek_tabungan->execute();
                $result_tabungan = $stmt_cek_tabungan->get_result();
                
                if ($result_tabungan && $result_tabungan->num_rows > 0) {
                    // Update existing record - tambahkan ke setoran dan saldo yang sudah ada
                    $data_tabungan = $result_tabungan->fetch_assoc();
                    $setoran_baru = floatval($data_tabungan['setoran']) + $nominal_simpanan;
                    $saldo_baru = floatval($data_tabungan['saldo']) + $nominal_simpanan;
                    
                    $stmt_update_tabungan->bind_param("ddsi", 
                        $setoran_baru, 
                        $saldo_baru, 
                        $tanggal_setor, 
                        $data_tabungan['id_tabungan']
                    );
                    
                    if (!$stmt_update_tabungan->execute()) {
                        throw new Exception("Error updating tabungan_usipa untuk anggota ID " . $anggota['id_anggota'] . ": " . $stmt_update_tabungan->error);
                    }
                } else {
                    // Insert new record - saldo sama dengan setoran karena baru pertama kali
                    $stmt_insert_tabungan->bind_param("iiidds", 
                        $anggota['id_anggota'], 
                        $tahun, 
                        $bulan, 
                        $nominal_simpanan,
                        $nominal_simpanan, 
                        $tanggal_setor
                    );
                    
                    if (!$stmt_insert_tabungan->execute()) {
                        throw new Exception("Error inserting tabungan_usipa untuk anggota ID " . $anggota['id_anggota'] . ": " . $stmt_insert_tabungan->error);
                    }
                }
                
                // ========== AKHIR INSERT/UPDATE TABUNGAN_USIPA ==========
                
                $total_dipotong += $nominal_simpanan;
                $jumlah_anggota++;
            }
            
            // ========== INSERT/UPDATE KE TABEL DANA_UNIT ==========
            
            // Cek apakah sudah ada record dana_unit untuk bulan dan tahun ini
            $stmt_cek_dana = $conn->prepare("SELECT id_dana, total_dana FROM dana_unit 
                WHERE id_unit = ? AND tahun = ? AND bulan = ?");
            
            if (!$stmt_cek_dana) {
                throw new Exception("Prepare cek dana_unit gagal: " . $conn->error);
            }
            
            $stmt_cek_dana->bind_param("iii", $id_unit, $tahun, $bulan);
            $stmt_cek_dana->execute();
            $result_dana = $stmt_cek_dana->get_result();
            
            $keterangan_dana = "Simpanan Masuk bulan " . $bulan_nama[$bulan] . " " . $tahun . " - Total dari " . $jumlah_anggota . " anggota";
            
            if ($result_dana && $result_dana->num_rows > 0) {
                // Update existing record - tambahkan ke total dana yang sudah ada
                $data_dana = $result_dana->fetch_assoc();
                $total_dana_baru = floatval($data_dana['total_dana']) + $total_dipotong;
                
                $stmt_update_dana = $conn->prepare("UPDATE dana_unit SET 
                    total_dana = ?, 
                    keterangan = ?,
                    tanggal_input = ?
                    WHERE id_dana = ?");
                
                if (!$stmt_update_dana) {
                    throw new Exception("Prepare update dana_unit gagal: " . $conn->error);
                }
                
                $stmt_update_dana->bind_param("dssi", $total_dana_baru, $keterangan_dana, $tanggal_setor, $data_dana['id_dana']);
                
                if (!$stmt_update_dana->execute()) {
                    throw new Exception("Error updating dana_unit: " . $stmt_update_dana->error);
                }
                
                $stmt_update_dana->close();
            } else {
                // Insert new record
                $stmt_insert_dana = $conn->prepare("INSERT INTO dana_unit 
                    (id_unit, tahun, bulan, total_dana, keterangan, tanggal_input) 
                    VALUES (?, ?, ?, ?, ?, ?)");
                
                if (!$stmt_insert_dana) {
                    throw new Exception("Prepare insert dana_unit gagal: " . $conn->error);
                }
                
                $stmt_insert_dana->bind_param("iiidss", 
                    $id_unit, 
                    $tahun, 
                    $bulan, 
                    $total_dipotong, 
                    $keterangan_dana, 
                    $tanggal_setor
                );
                
                if (!$stmt_insert_dana->execute()) {
                    throw new Exception("Error inserting dana_unit: " . $stmt_insert_dana->error);
                }
                
                $stmt_insert_dana->close();
            }
            
            $stmt_cek_dana->close();
            
            // ========== AKHIR INSERT/UPDATE DANA_UNIT ==========
            
            // Commit transaksi
            $conn->commit();
            
            // Close statements
            $stmt_simpanan->close();
            $stmt_cek_tabungan->close();
            $stmt_update_tabungan->close();
            $stmt_insert_tabungan->close();
            $stmt_anggota->close();
            
            header("Location: potong_gaji.php?success=1&total=$total_dipotong&jumlah=$jumlah_anggota&periode=" . urlencode($bulan_nama[$bulan] . " " . $tahun));
            exit;
            
        } catch (Exception $e) {
            $conn->rollback();
            error_log("Error potong gaji: " . $e->getMessage());
            header("Location: potong_gaji.php?error=gagal&msg=" . urlencode($e->getMessage()));
            exit;
        }
    } else {
        $stmt_anggota->close();
        header("Location: potong_gaji.php?error=no_anggota");
        exit;
    }
}

// Ambil riwayat pemotongan
$riwayat = $conn->query("SELECT 
    YEAR(tanggal_setor) as tahun,
    MONTH(tanggal_setor) as bulan,
    COUNT(DISTINCT id_anggota) as jumlah_anggota,
    SUM(nominal) as total_nominal,
    MIN(tanggal_setor) as tanggal_proses
    FROM simpanan 
    WHERE jenis = 'wajib' AND (keterangan LIKE '%Pemotongan Gaji Otomatis%' OR keterangan LIKE '%Potong Gaji%')
    GROUP BY YEAR(tanggal_setor), MONTH(tanggal_setor)
    ORDER BY tahun DESC, bulan DESC
    LIMIT 12");

// Cek bulan-bulan yang sudah dipotong (untuk validasi frontend)
$bulan_sudah_potong = [];
$stmt_bulan = $conn->query("SELECT DISTINCT YEAR(tanggal_setor) as tahun, MONTH(tanggal_setor) as bulan 
    FROM simpanan 
    WHERE jenis = 'wajib' AND (keterangan LIKE '%Pemotongan Gaji Otomatis%' OR keterangan LIKE '%Potong Gaji%')");

if ($stmt_bulan) {
    while ($row = $stmt_bulan->fetch_assoc()) {
        $bulan_sudah_potong[] = $row['tahun'] . '-' . str_pad($row['bulan'], 2, '0', STR_PAD_LEFT);
    }
}
?>

<!DOCTYPE html>
<html lang="id">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Pemotongan Gaji - Koperasi</title>
    <script src="https://cdn.tailwindcss.com"></script>
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.0.0/css/all.min.css">
    <script>
        tailwind.config = {
            theme: {
                extend: {
                    colors: {
                        primary: {
                            50: '#f5f7ff',
                            500: '#667eea',
                            600: '#5568d3',
                        },
                        purple: {
                            500: '#764ba2',
                        }
                    }
                }
            }
        }
    </script>
</head>
<body class="bg-gradient-to-br from-red-500 via-pink-500 to-purple-600 min-h-screen">
    
    <!-- Sidebar -->
    <aside class="fixed top-0 left-0 h-screen w-64 bg-white shadow-2xl z-50 overflow-y-auto">
        <div class="p-6 border-b border-gray-200 bg-gradient-to-r from-red-500 to-pink-500">
            <div class="flex items-center space-x-3 mb-4">
                <img src="img/logo.png" alt="Logo" class="w-16 h-16 object-contain bg-white rounded-xl p-2 shadow-lg">
                <div>
                    <h2 class="text-white font-bold text-lg">Koperasi</h2>
                    <p class="text-red-100 text-xs">Unit <?= htmlspecialchars($nama_unit) ?></p>
                </div>
            </div>
            <div class="pt-3 border-t border-red-400">
                <div class="flex items-center space-x-3">
                    <div class="w-10 h-10 bg-white rounded-lg flex items-center justify-center shadow">
                        <i class="fas fa-user-tie text-red-600 text-xl"></i>
                    </div>
                    <div>
                        <h3 class="text-white font-bold text-sm">Kepala Unit <?= htmlspecialchars($nama_unit) ?></h3>
                        <p class="text-red-100 text-xs">Pemotongan Gaji</p>
                    </div>
                </div>
            </div>
        </div>

        <nav class="p-4 space-y-2">
            <a href="dashboard_kepalaunit_toko.php?menu=approval_pinjaman" class="flex items-center space-x-3 p-3 rounded-lg text-gray-700 hover:bg-gray-100">
                <i class="fas fa-hand-holding-usd w-5"></i>
                <span class="font-medium">Approval Pinjaman</span>
            </a>
            
            <a href="potong_gaji.php" class="flex items-center space-x-3 p-3 rounded-lg bg-gradient-to-r from-red-500 to-pink-500 text-white">
                <i class="fas fa-cut w-5"></i>
                <span class="font-medium">Potong Gaji</span>
            </a>
            
            <a href="input_simpanan_sukarela.php" class="flex items-center space-x-3 p-3 rounded-lg text-gray-700 hover:bg-gray-100">
                <i class="fas fa-hand-holding-heart w-5"></i>
                <span class="font-medium">Simpanan Sukarela</span>
            </a>
            
           <a href="dana_unit.php" class="flex items-center space-x-3 p-3 rounded-lg text-gray-700 hover:bg-gray-100">
                <i class="fas fa-money-bill-wave w-5"></i>
                <span class="font-medium">Dana Unit</span>
            </a>

             <a href="konfigurasi_shu.php" class="flex items-center space-x-3 p-3 rounded-lg text-gray-700 hover:bg-gray-100">
                <i class="fas fa-chart-pie w-5"></i>
                <span class="font-medium">Konfigurasi SHU</span>
            </a>

            <div class="border-t border-gray-200 my-4"></div>

            <a href="logout.php" class="flex items-center space-x-3 p-3 rounded-lg text-red-600 hover:bg-red-50">
                <i class="fas fa-sign-out-alt w-5"></i>
                <span class="font-medium">Logout</span>
            </a>
        </nav>
    </aside>
            
    <!-- Main Content -->
    <main class="ml-64 p-8">
        <div class="max-w-7xl mx-auto">
            
            <!-- Header -->
            <div class="mb-8">
                <h1 class="text-4xl font-bold text-white mb-2">
                    <i class="fas fa-cut mr-3"></i>Pemotongan Gaji Otomatis
                </h1>
                <p class="text-red-100">Potong gaji semua anggota untuk simpanan wajib</p>
            </div>

            <?php if (isset($_GET['success'])): ?>
            <div class="mb-6 bg-green-100 border border-green-400 text-green-700 px-4 py-3 rounded-lg shadow">
                <i class="fas fa-check-circle mr-2"></i>
                Berhasil memotong gaji <?= htmlspecialchars($_GET['jumlah']) ?> anggota 
                periode <?= htmlspecialchars($_GET['periode'] ?? '') ?> 
                dengan total Rp <?= number_format($_GET['total'], 0, ',', '.') ?>!
                <br><small class="text-green-600">Data telah masuk ke tabel Simpanan, Tabungan USIPA, dan Dana Unit.</small>
            </div>
            <?php endif; ?>

            <?php if (isset($_GET['error'])): ?>
            <div class="mb-6 bg-red-100 border border-red-400 text-red-700 px-4 py-3 rounded-lg shadow">
                <i class="fas fa-exclamation-circle mr-2"></i>
                <?php if ($_GET['error'] == 'sudah_potong'): ?>
                    Pemotongan gaji untuk bulan dan tahun tersebut sudah pernah dilakukan!
                <?php elseif ($_GET['error'] == 'bulan_belum_tiba'): ?>
                    Anda tidak bisa memotong gaji untuk bulan yang belum tiba! Harap pilih bulan yang sudah lewat atau bulan ini.
                <?php elseif ($_GET['error'] == 'gagal'): ?>
                    Terjadi kesalahan saat memproses pemotongan gaji!
                    <?php if (isset($_GET['msg'])): ?>
                        <br><small>Detail: <?= htmlspecialchars($_GET['msg']) ?></small>
                    <?php endif; ?>
                <?php elseif ($_GET['error'] == 'no_anggota'): ?>
                    Tidak ada anggota aktif dengan gaji pokok yang ditemukan!
                <?php else: ?>
                    Terjadi kesalahan!
                <?php endif; ?>
            </div>
            <?php endif; ?>

            <!-- Form Pemotongan -->
            <div class="bg-white rounded-2xl shadow-2xl overflow-hidden mb-8">
                <div class="bg-gradient-to-r from-red-500 to-pink-500 p-6">
                    <h2 class="text-2xl font-bold text-white">
                        <i class="fas fa-calculator mr-3"></i>Form Pemotongan Gaji
                    </h2>
                    <p class="text-red-100 text-sm mt-1">Pilih periode untuk melakukan pemotongan gaji</p>
                </div>
                
                <div class="p-6">
                    <div class="bg-gradient-to-br from-red-50 to-pink-50 border border-red-200 rounded-xl p-6">
                        <form method="POST" id="formPotongGaji" onsubmit="return validateForm()">
                            <div class="grid grid-cols-1 md:grid-cols-2 gap-4 mb-6">
                                <div>
                                    <label class="block text-sm font-medium text-gray-700 mb-2">Tahun</label>
                                    <select name="tahun_potong" id="tahun_potong" required class="w-full px-4 py-3 border rounded-lg focus:ring-2 focus:ring-red-500" onchange="checkPeriode()">
                                        <?php for($y = date('Y'); $y >= 2020; $y--): ?>
                                        <option value="<?= $y ?>" <?= $y == date('Y') ? 'selected' : '' ?>><?= $y ?></option>
                                        <?php endfor; ?>
                                    </select>
                                </div>
                                
                                <div>
                                    <label class="block text-sm font-medium text-gray-700 mb-2">Bulan</label>
                                    <select name="bulan_potong" id="bulan_potong" required class="w-full px-4 py-3 border rounded-lg focus:ring-2 focus:ring-red-500" onchange="checkPeriode()">
                                        <?php for($m = 1; $m <= 12; $m++): ?>
                                        <option value="<?= $m ?>" <?= $m == date('n') ? 'selected' : '' ?>><?= $bulan_nama[$m] ?></option>
                                        <?php endfor; ?>
                                    </select>
                                </div>
                            </div>
                            
                            <div id="alertSudahPotong" class="bg-red-100 border border-red-300 rounded-lg p-4 mb-6 hidden">
                                <p class="text-sm text-red-800 font-semibold">
                                    <i class="fas fa-exclamation-triangle mr-2"></i>
                                    Bulan dan tahun ini sudah pernah dipotong! Silakan pilih periode lain.
                                </p>
                            </div>
                            
                            <div id="alertBelumTiba" class="bg-yellow-100 border border-yellow-300 rounded-lg p-4 mb-6 hidden">
                                <p class="text-sm text-yellow-800 font-semibold">
                                    <i class="fas fa-clock mr-2"></i>
                                    Bulan yang dipilih belum tiba! Anda hanya bisa memotong gaji untuk bulan yang sudah lewat atau bulan ini.
                                </p>
                            </div>
                            
                            <div class="bg-blue-50 border border-blue-200 rounded-lg p-4 mb-6">
                                <h4 class="font-semibold text-blue-800 mb-2">
                                    <i class="fas fa-info-circle mr-2"></i>Informasi Penting:
                                </h4>
                                <ul class="text-sm text-blue-700 space-y-1">
                                    <li>• <strong>Nominal simpanan wajib adalah Rp 50.000 per anggota</strong></li>
                                    <li>• Pemotongan hanya bisa dilakukan sekali per bulan</li>
                                    <li>• Hanya bisa memotong gaji untuk bulan yang sudah lewat atau bulan berjalan</li>
                                    <li>• Data akan otomatis masuk ke tabel Simpanan</li>
                                    <li>• <strong>Data akan otomatis masuk ke tabel Tabungan USIPA</strong></li>
                                    <li>• <strong>Total pemotongan akan otomatis masuk ke Dana Unit</strong></li>
                                    <li>• Pastikan periode yang dipilih sudah benar</li>
                                </ul>
                            </div>
                            
                            <button type="submit" name="potong_gaji_otomatis" id="btnSubmit" class="w-full bg-gradient-to-r from-red-500 to-pink-500 text-white px-6 py-4 rounded-lg font-semibold hover:shadow-lg transition text-lg">
                                <i class="fas fa-cut mr-2"></i>Proses Pemotongan Gaji
                            </button>
                        </form>
                    </div>
                </div>
            </div>

            <!-- Riwayat Pemotongan -->
            <div class="bg-white rounded-2xl shadow-2xl overflow-hidden">
                <div class="bg-gradient-to-r from-purple-500 to-indigo-500 p-6">
                    <h2 class="text-2xl font-bold text-white">
                        <i class="fas fa-history mr-3"></i>Riwayat Pemotongan Gaji
                    </h2>
                    <p class="text-purple-100 text-sm mt-1">12 pemotongan terakhir</p>
                </div>
                
                <div class="p-6">
                    <div class="overflow-x-auto">
                        <table class="w-full">
                            <thead>
                                <tr class="bg-gray-50 border-b-2 border-purple-500">
                                    <th class="px-6 py-4 text-left text-xs font-bold text-gray-700 uppercase">Periode</th>
                                    <th class="px-6 py-4 text-right text-xs font-bold text-gray-700 uppercase">Jumlah Anggota</th>
                                    <th class="px-6 py-4 text-right text-xs font-bold text-gray-700 uppercase">Total Dipotong</th>
                                    <th class="px-6 py-4 text-left text-xs font-bold text-gray-700 uppercase">Tanggal Proses</th>
                                </tr>
                            </thead>
                            <tbody class="divide-y divide-gray-200">
                                <?php if ($riwayat && $riwayat->num_rows > 0): ?>
                                    <?php while ($row = $riwayat->fetch_assoc()): ?>
                                    <tr class="hover:bg-gray-50">
                                        <td class="px-6 py-4 font-semibold"><?= htmlspecialchars($bulan_nama[$row['bulan']]) ?> <?= htmlspecialchars($row['tahun']) ?></td>
                                        <td class="px-6 py-4 text-right"><?= number_format($row['jumlah_anggota'], 0, ',', '.') ?> anggota</td>
                                        <td class="px-6 py-4 text-right">
                                            <span class="text-lg font-bold text-red-600">Rp <?= number_format($row['total_nominal'], 0, ',', '.') ?></span>
                                        </td>
                                        <td class="px-6 py-4"><?= date('d/m/Y', strtotime($row['tanggal_proses'])) ?></td>
                                    </tr>
                                    <?php endwhile; ?>
                                <?php else: ?>
                                    <tr>
                                        <td colspan="4" class="px-6 py-12 text-center text-gray-500">
                                            <i class="fas fa-inbox text-6xl mb-4 text-gray-300"></i>
                                            <p>Belum ada riwayat pemotongan gaji</p>
                                        </td>
                                    </tr>
                                <?php endif; ?>
                            </tbody>
                        </table>
                    </div>
                </div>
            </div>

        </div>
    </main>

    <script>
        // Data bulan yang sudah dipotong dari PHP
        const bulanSudahPotong = <?= json_encode($bulan_sudah_potong) ?>;
        
        function checkPeriode() {
            const tahun = document.getElementById('tahun_potong').value;
            const bulan = document.getElementById('bulan_potong').value;
            const periode = tahun + '-' + bulan.padStart(2, '0');
            
            const alertSudahPotong = document.getElementById('alertSudahPotong');
            const alertBelumTiba = document.getElementById('alertBelumTiba');
            const btnSubmit = document.getElementById('btnSubmit');
            
            // Reset alerts
            alertSudahPotong.classList.add('hidden');
            alertBelumTiba.classList.add('hidden');
            btnSubmit.disabled = false;
            btnSubmit.classList.remove('opacity-50', 'cursor-not-allowed');
            
            // Cek apakah sudah pernah dipotong
            if (bulanSudahPotong.includes(periode)) {
                alertSudahPotong.classList.remove('hidden');
                btnSubmit.disabled = true;
                btnSubmit.classList.add('opacity-50', 'cursor-not-allowed');
                return false;
            }
            
            // Cek apakah bulan belum tiba
            const sekarang = new Date();
            const bulanSekarang = sekarang.getMonth() + 1;
            const tahunSekarang = sekarang.getFullYear();
            
            if (parseInt(tahun) > tahunSekarang || 
                (parseInt(tahun) === tahunSekarang && parseInt(bulan) > bulanSekarang)) {
                alertBelumTiba.classList.remove('hidden');
                btnSubmit.disabled = true;
                btnSubmit.classList.add('opacity-50', 'cursor-not-allowed');
                return false;
            }
            
            return true;
        }
        
        function validateForm() {
            if (!checkPeriode()) {
                return false;
            }
            
            return confirm('Anda yakin akan melakukan pemotongan gaji otomatis sebesar Rp 50.000 per anggota?\n\nProses ini akan:\n• Menambah data ke tabel Simpanan\n• Menambah data ke tabel Tabungan USIPA\n• Menambah Dana Unit\n\nProses ini tidak dapat dibatalkan!');
        }
        
        // Jalankan pengecekan saat halaman dimuat
        window.addEventListener('DOMContentLoaded', checkPeriode);
    </script>

</body>
</html>