<?php
session_start();
include 'config.php';

// Cek apakah user sudah login dan memiliki role Kasir Defmart
if (!isset($_SESSION['role']) || $_SESSION['role'] !== 'Kasir Defmart') {
    header("Location: index.php");
    exit;
}

$id_user = $_SESSION['id_user'];
$username = $_SESSION['username'];

// Hapus Transaksi
if (isset($_GET['hapus_transaksi'])) {
    $id_transaksi = intval($_GET['hapus_transaksi']);
    
    // Mulai transaksi database
    $conn->begin_transaction();
    
    try {
        // Ambil data transaksi terlebih dahulu
        $stmt_get = $conn->prepare("SELECT id_anggota, total_harga, metode_pembayaran, tanggal_transaksi FROM defmart_transaksi WHERE id_transaksi = ?");
        if (!$stmt_get) {
            throw new Exception("Prepare statement gagal: " . $conn->error);
        }
        $stmt_get->bind_param("i", $id_transaksi);
        $stmt_get->execute();
        $result_get = $stmt_get->get_result();
        $transaksi_data = $result_get->fetch_assoc();
        $stmt_get->close();
        
        if ($transaksi_data) {
            // Hapus dari defmart_transaksi
            $stmt = $conn->prepare("DELETE FROM defmart_transaksi WHERE id_transaksi = ?");
            if (!$stmt) {
                throw new Exception("Prepare statement gagal: " . $conn->error);
            }
            $stmt->bind_param("i", $id_transaksi);
            $stmt->execute();
            $stmt->close();
            
            // Kurangi total_dana_masuk di tabel unit untuk Defmart
            $stmt_unit = $conn->prepare("UPDATE unit SET total_dana_masuk = total_dana_masuk - ? WHERE nama_unit = 'Defmart'");
            if (!$stmt_unit) {
                throw new Exception("Prepare statement gagal: " . $conn->error);
            }
            $stmt_unit->bind_param("d", $transaksi_data['total_harga']);
            $stmt_unit->execute();
            $stmt_unit->close();
            
            // Get id_unit untuk Defmart
            $stmt_get_unit = $conn->prepare("SELECT id_unit FROM unit WHERE nama_unit = 'Defmart'");
            if (!$stmt_get_unit) {
                throw new Exception("Prepare statement gagal: " . $conn->error);
            }
            $stmt_get_unit->execute();
            $result_unit = $stmt_get_unit->get_result();
            $unit_data = $result_unit->fetch_assoc();
            
            if (!$unit_data) {
                throw new Exception("Unit Defmart tidak ditemukan");
            }
            
            $id_unit = $unit_data['id_unit'];
            $stmt_get_unit->close();
            
            // Update tabel dana_unit
            $tahun = date('Y', strtotime($transaksi_data['tanggal_transaksi']));
            $bulan = date('n', strtotime($transaksi_data['tanggal_transaksi']));
            
            $stmt_update_dana = $conn->prepare("UPDATE dana_unit SET total_dana = total_dana - ? WHERE id_unit = ? AND tahun = ? AND bulan = ?");
            if (!$stmt_update_dana) {
                throw new Exception("Prepare statement gagal: " . $conn->error);
            }
            $stmt_update_dana->bind_param("diii", $transaksi_data['total_harga'], $id_unit, $tahun, $bulan);
            $stmt_update_dana->execute();
            $stmt_update_dana->close();
            
            // Jika pembayaran dengan Bon, kurangi dari potongan_gaji
            if ($transaksi_data['metode_pembayaran'] === 'Bon') {
                $bulan_periode = date('Y-m-01', strtotime($transaksi_data['tanggal_transaksi']));
                
                // Cek potongan yang ada
                $stmt_check_potongan = $conn->prepare("SELECT id_potongan, jumlah FROM potongan_gaji WHERE id_anggota = ? AND sumber = 'Defmart' AND bulan_periode = ? AND status = 'pending'");
                if (!$stmt_check_potongan) {
                    throw new Exception("Prepare statement gagal: " . $conn->error);
                }
                $stmt_check_potongan->bind_param("is", $transaksi_data['id_anggota'], $bulan_periode);
                $stmt_check_potongan->execute();
                $result_potongan = $stmt_check_potongan->get_result();
                
                if ($result_potongan->num_rows > 0) {
                    $potongan_data = $result_potongan->fetch_assoc();
                    $jumlah_baru = $potongan_data['jumlah'] - $transaksi_data['total_harga'];
                    
                    if ($jumlah_baru <= 0) {
                        // Hapus record jika jumlah menjadi 0 atau kurang
                        $stmt_hapus_potongan = $conn->prepare("DELETE FROM potongan_gaji WHERE id_potongan = ?");
                        if (!$stmt_hapus_potongan) {
                            throw new Exception("Prepare statement gagal: " . $conn->error);
                        }
                        $stmt_hapus_potongan->bind_param("i", $potongan_data['id_potongan']);
                        $stmt_hapus_potongan->execute();
                        $stmt_hapus_potongan->close();
                    } else {
                        // Update jumlah potongan
                        $stmt_update_potongan = $conn->prepare("UPDATE potongan_gaji SET jumlah = ? WHERE id_potongan = ?");
                        if (!$stmt_update_potongan) {
                            throw new Exception("Prepare statement gagal: " . $conn->error);
                        }
                        $stmt_update_potongan->bind_param("di", $jumlah_baru, $potongan_data['id_potongan']);
                        $stmt_update_potongan->execute();
                        $stmt_update_potongan->close();
                    }
                }
                
                $stmt_check_potongan->close();
            }
            
            // Commit transaksi
            $conn->commit();
            $sukses = "Transaksi berhasil dihapus!";
            
        } else {
            $conn->rollback();
            $error = "Transaksi tidak ditemukan!";
        }
        
    } catch (Exception $e) {
        // Rollback jika ada error
        $conn->rollback();
        $error = "Gagal menghapus transaksi: " . $e->getMessage();
    }
}

// Ambil data anggota untuk dropdown
$anggota_result = $conn->query("SELECT id_anggota, nama FROM anggota ORDER BY nama");

// Filter dan Sorting
$filter_bulan = isset($_GET['filter_bulan']) ? $_GET['filter_bulan'] : date('Y-m');
$filter_anggota = isset($_GET['filter_anggota']) ? $_GET['filter_anggota'] : '';
$sort_by = isset($_GET['sort_by']) ? $_GET['sort_by'] : 'tanggal';
$urutan = isset($_GET['urutan']) ? $_GET['urutan'] : 'DESC';

// Ambil tahun untuk laporan SHU
$filter_tahun_shu = isset($_GET['filter_tahun_shu']) ? $_GET['filter_tahun_shu'] : date('Y');

// Ambil data transaksi dengan filter
$query_transaksi = "SELECT dt.*, a.nama FROM defmart_transaksi dt 
                    LEFT JOIN anggota a ON dt.id_anggota = a.id_anggota 
                    WHERE DATE_FORMAT(dt.tanggal_transaksi, '%Y-%m') = ?";
$params_types = "s";
$params = [$filter_bulan];

// Tambahkan filter anggota jika dipilih
if ($filter_anggota !== '') {
    $query_transaksi .= " AND dt.id_anggota = ?";
    $params_types .= "i";
    $params[] = intval($filter_anggota);
}

if ($sort_by === 'tanggal') {
    $query_transaksi .= " ORDER BY dt.tanggal_transaksi " . $urutan;
} else if ($sort_by === 'total') {
    $query_transaksi .= " ORDER BY dt.total_harga " . $urutan;
}

$query_transaksi .= " LIMIT 100";

$stmt_transaksi = $conn->prepare($query_transaksi);
if (!$stmt_transaksi) {
    die("Prepare statement gagal: " . $conn->error);
}
$stmt_transaksi->bind_param($params_types, ...$params);
$stmt_transaksi->execute();
$transaksi_result = $stmt_transaksi->get_result();

// Query untuk laporan agregat SHU Defmart per anggota
$query_shu = "SELECT 
                a.id_anggota,
                a.nama,
                a.nip,
                COALESCE(SUM(dt.total_harga), 0) as total_transaksi,
                COUNT(dt.id_transaksi) as jumlah_transaksi,
                COALESCE(sp.shu_toko, 0) as shu_toko_tahun_ini,
                COALESCE(sp.total_shu, 0) as total_shu_tahun_ini
            FROM anggota a
            LEFT JOIN defmart_transaksi dt ON a.id_anggota = dt.id_anggota 
                AND YEAR(dt.tanggal_transaksi) = ?
            LEFT JOIN shu_pembagian sp ON a.id_anggota = sp.id_anggota 
                AND sp.tahun = ?
            GROUP BY a.id_anggota, a.nama, a.nip, sp.shu_toko, sp.total_shu
            HAVING total_transaksi > 0
            ORDER BY total_transaksi DESC";

$stmt_shu = $conn->prepare($query_shu);
$stmt_shu->bind_param("ii", $filter_tahun_shu, $filter_tahun_shu);
$stmt_shu->execute();
$shu_result = $stmt_shu->get_result();

// Hitung total keseluruhan
$query_total = "SELECT 
                    COALESCE(SUM(total_harga), 0) as grand_total_transaksi,
                    COUNT(id_transaksi) as grand_jumlah_transaksi,
                    COUNT(DISTINCT id_anggota) as total_anggota_aktif
                FROM defmart_transaksi 
                WHERE YEAR(tanggal_transaksi) = ?";
$stmt_total = $conn->prepare($query_total);
$stmt_total->bind_param("i", $filter_tahun_shu);
$stmt_total->execute();
$total_result = $stmt_total->get_result();
$total_data = $total_result->fetch_assoc();
?>

<!DOCTYPE html>
<html lang="id">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Riwayat Transaksi - Koperasi PUSDATIN</title>
    <script src="https://cdn.tailwindcss.com"></script>
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.0.0/css/all.min.css">
    <style>
        .gradient-primary { background: linear-gradient(135deg, #667eea 0%, #764ba2 100%); }
        .gradient-success { background: linear-gradient(135deg, #34d399 0%, #059669 100%); }
        .gradient-warning { background: linear-gradient(135deg, #fbbf24 0%, #f59e0b 100%); }
        .gradient-info { background: linear-gradient(135deg, #60a5fa 0%, #3b82f6 100%); }
        .tab-button.active {
            border-bottom: 3px solid #667eea;
            color: #667eea;
        }
    </style>
</head>
<body class="bg-gray-100">

<!-- Sidebar -->
<?php include 'sidebar_kasir.php'; ?>

<!-- Main Content -->
<div class="ml-64 min-h-screen">
    <!-- Top Bar -->
    <div class="bg-white shadow-sm sticky top-0 z-30">
        <div class="px-6 py-4">
            <h1 class="text-2xl font-bold text-gray-800">
                <i class="fas fa-history mr-2 text-purple-500"></i>Riwayat Transaksi & Laporan SHU
            </h1>
        </div>
    </div>

    <!-- Content Area -->
    <div class="p-6">
        <!-- Alert Messages -->
        <?php if (isset($sukses)): ?>
            <div class="mb-4 p-4 bg-green-50 border border-green-200 rounded-lg flex items-start">
                <i class="fas fa-check-circle text-green-500 mt-1 mr-3"></i>
                <p class="text-green-800 font-semibold"><?= $sukses ?></p>
            </div>
        <?php endif; ?>

        <?php if (isset($error)): ?>
            <div class="mb-4 p-4 bg-red-50 border border-red-200 rounded-lg flex items-start">
                <i class="fas fa-exclamation-circle text-red-500 mt-1 mr-3"></i>
                <p class="text-red-800 font-semibold"><?= $error ?></p>
            </div>
        <?php endif; ?>

        <!-- Tab Navigation -->
        <div class="bg-white rounded-t-xl shadow-sm mb-0">
            <div class="flex border-b border-gray-200">
                <button onclick="showTab('riwayat')" id="tab-riwayat" class="tab-button active px-6 py-4 font-semibold text-gray-700 hover:text-purple-600 transition">
                    <i class="fas fa-list mr-2"></i>Riwayat Transaksi
                </button>
                <button onclick="showTab('shu')" id="tab-shu" class="tab-button px-6 py-4 font-semibold text-gray-700 hover:text-purple-600 transition">
                    <i class="fas fa-chart-line mr-2"></i>Laporan SHU Defmart
                </button>
            </div>
        </div>

        <!-- Tab Content: Riwayat Transaksi -->
        <div id="content-riwayat" class="tab-content">
            <!-- Filter Section -->
            <div class="bg-white shadow-sm p-6 mb-6">
                <h3 class="text-lg font-semibold text-gray-800 mb-4">
                    <i class="fas fa-filter mr-2 text-blue-500"></i>Filter & Pengurutan
                </h3>
                <div class="grid grid-cols-1 md:grid-cols-5 gap-4">
                    <div>
                        <label class="block text-sm font-medium text-gray-700 mb-2">Bulan</label>
                        <input type="month" id="filter_bulan" class="w-full px-4 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-transparent" value="<?= $filter_bulan ?>">
                    </div>
                    <div>
                        <label class="block text-sm font-medium text-gray-700 mb-2">Anggota</label>
                        <select id="filter_anggota" class="w-full px-4 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-transparent">
                            <option value="">Semua Anggota</option>
                            <?php 
                            $anggota_result->data_seek(0);
                            while ($anggota = $anggota_result->fetch_assoc()): 
                            ?>
                                <option value="<?= $anggota['id_anggota'] ?>" <?= $filter_anggota == $anggota['id_anggota'] ? 'selected' : '' ?>>
                                    <?= htmlspecialchars($anggota['nama']) ?>
                                </option>
                            <?php endwhile; ?>
                        </select>
                    </div>
                    <div>
                        <label class="block text-sm font-medium text-gray-700 mb-2">Urutkan</label>
                        <select id="sort_by" class="w-full px-4 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-transparent">
                            <option value="tanggal" <?= $sort_by === 'tanggal' ? 'selected' : '' ?>>Tanggal</option>
                            <option value="total" <?= $sort_by === 'total' ? 'selected' : '' ?>>Total Harga</option>
                        </select>
                    </div>
                    <div>
                        <label class="block text-sm font-medium text-gray-700 mb-2">Urutan</label>
                        <select id="urutan" class="w-full px-4 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-transparent">
                            <option value="DESC" <?= $urutan === 'DESC' ? 'selected' : '' ?>>Terbaru/Terbesar</option>
                            <option value="ASC" <?= $urutan === 'ASC' ? 'selected' : '' ?>>Terlama/Terkecil</option>
                        </select>
                    </div>
                    <div>
                        <label class="block text-sm font-medium text-gray-700 mb-2">&nbsp;</label>
                        <button onclick="applyFilter()" class="w-full gradient-primary text-white font-semibold py-2 rounded-lg hover:shadow-lg transition">
                            <i class="fas fa-filter mr-2"></i>Terapkan
                        </button>
                    </div>
                </div>
            </div>

            <!-- Tabel Transaksi -->
            <div class="bg-white rounded-xl shadow-sm overflow-hidden">
                <div class="p-6 border-b border-gray-200 flex justify-between items-center">
                    <h3 class="text-lg font-semibold text-gray-800">
                        Data Transaksi
                        <?php if ($filter_anggota !== ''): ?>
                            <?php
                            $anggota_result->data_seek(0);
                            while ($anggota = $anggota_result->fetch_assoc()) {
                                if ($anggota['id_anggota'] == $filter_anggota) {
                                    echo '<span class="text-sm text-gray-600 ml-2">- ' . htmlspecialchars($anggota['nama']) . '</span>';
                                    break;
                                }
                            }
                            ?>
                        <?php endif; ?>
                    </h3>
                    <div class="flex gap-2">
                        <a href="export_transaksi_kasir.php?filter_bulan=<?= $filter_bulan ?>&filter_anggota=<?= $filter_anggota ?>&sort_by=<?= $sort_by ?>&urutan=<?= $urutan ?>" 
                           class="px-4 py-2 bg-green-600 hover:bg-green-700 text-white rounded-lg font-semibold transition text-sm">
                            <i class="fas fa-file-excel mr-2"></i>Excel
                        </a>
                        <a href="export_transaksi_pdf.php?filter_bulan=<?= $filter_bulan ?>&filter_anggota=<?= $filter_anggota ?>&sort_by=<?= $sort_by ?>&urutan=<?= $urutan ?>" 
                           target="_blank"
                           class="px-4 py-2 bg-red-600 hover:bg-red-700 text-white rounded-lg font-semibold transition text-sm">
                            <i class="fas fa-file-pdf mr-2"></i>PDF
                        </a>
                    </div>
                </div>
                <div class="overflow-x-auto">
                    <table class="w-full">
                        <thead>
                            <tr class="bg-gray-50 border-b border-gray-200">
                                <th class="px-6 py-3 text-left text-sm font-semibold text-gray-700">No</th>
                                <th class="px-6 py-3 text-left text-sm font-semibold text-gray-700">Tanggal</th>
                                <th class="px-6 py-3 text-left text-sm font-semibold text-gray-700">Anggota</th>
                                <th class="px-6 py-3 text-left text-sm font-semibold text-gray-700">Barang</th>
                                <th class="px-6 py-3 text-left text-sm font-semibold text-gray-700">Qty</th>
                                <th class="px-6 py-3 text-left text-sm font-semibold text-gray-700">Harga Satuan</th>
                                <th class="px-6 py-3 text-left text-sm font-semibold text-gray-700">Total</th>
                                <th class="px-6 py-3 text-left text-sm font-semibold text-gray-700">Pembayaran</th>
                                <th class="px-6 py-3 text-left text-sm font-semibold text-gray-700">Aksi</th>
                            </tr>
                        </thead>
                        <tbody class="divide-y divide-gray-200">
                            <?php 
                            $no = 1;
                            while ($row = $transaksi_result->fetch_assoc()): 
                            ?>
                                <tr class="hover:bg-gray-50 transition">
                                    <td class="px-6 py-4 text-sm font-semibold text-gray-900"><?= $no++ ?></td>
                                    <td class="px-6 py-4 text-sm text-gray-600"><?= date('d/m/Y H:i', strtotime($row['tanggal_transaksi'])) ?></td>
                                    <td class="px-6 py-4 text-sm text-gray-600"><?= htmlspecialchars($row['nama'] ?? '-') ?></td>
                                    <td class="px-6 py-4 text-sm text-gray-600"><?= htmlspecialchars($row['nama_barang']) ?></td>
                                    <td class="px-6 py-4 text-sm text-gray-600"><?= $row['qty'] ?></td>
                                    <td class="px-6 py-4 text-sm text-gray-600">Rp <?= number_format($row['harga_satuan'], 0, ',', '.') ?></td>
                                    <td class="px-6 py-4 text-sm font-semibold text-gray-900">Rp <?= number_format($row['total_harga'], 0, ',', '.') ?></td>
                                    <td class="px-6 py-4 text-sm">
                                        <?php if ($row['metode_pembayaran'] === 'Bon'): ?>
                                            <span class="px-3 py-1 rounded-full text-white text-xs font-semibold bg-orange-500">
                                                <i class="fas fa-receipt mr-1"></i>Bon
                                            </span>
                                        <?php else: ?>
                                            <span class="px-3 py-1 rounded-full text-white text-xs font-semibold gradient-primary">
                                                Cash
                                            </span>
                                        <?php endif; ?>
                                    </td>
                                    <td class="px-6 py-4 text-sm">
                                        <a href="?hapus_transaksi=<?= $row['id_transaksi'] ?>&filter_bulan=<?= $filter_bulan ?>&filter_anggota=<?= $filter_anggota ?>&sort_by=<?= $sort_by ?>&urutan=<?= $urutan ?>" 
                                           class="px-3 py-1 bg-red-100 text-red-700 rounded-lg hover:bg-red-200 transition text-xs font-semibold" 
                                           onclick="return confirm('Yakin hapus transaksi ini?')">
                                            <i class="fas fa-trash"></i> Hapus
                                        </a>
                                    </td>
                                </tr>
                            <?php endwhile; ?>
                        </tbody>
                    </table>
                    <?php if ($transaksi_result->num_rows == 0): ?>
                        <div class="p-6 text-center text-gray-500">
                            <i class="fas fa-inbox text-4xl mb-3 block text-gray-300"></i>
                            Tidak ada data transaksi untuk filter yang dipilih
                        </div>
                    <?php endif; ?>
                </div>
            </div>
        </div>

        <!-- Tab Content: Laporan SHU -->
        <div id="content-shu" class="tab-content hidden">
            <!-- Filter Tahun SHU -->
            <div class="bg-white shadow-sm p-6 mb-6">
                <h3 class="text-lg font-semibold text-gray-800 mb-4">
                    <i class="fas fa-calendar-alt mr-2 text-blue-500"></i>Filter Laporan SHU
                </h3>
                <div class="grid grid-cols-1 md:grid-cols-3 gap-4">
                    <div>
                        <label class="block text-sm font-medium text-gray-700 mb-2">Tahun</label>
                        <select id="filter_tahun_shu" class="w-full px-4 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-transparent">
                            <?php for($y = date('Y'); $y >= 2020; $y--): ?>
                                <option value="<?= $y ?>" <?= $filter_tahun_shu == $y ? 'selected' : '' ?>><?= $y ?></option>
                            <?php endfor; ?>
                        </select>
                    </div>
                    <div>
                        <label class="block text-sm font-medium text-gray-700 mb-2">&nbsp;</label>
                        <button onclick="applySHUFilter()" class="w-full gradient-primary text-white font-semibold py-2 rounded-lg hover:shadow-lg transition">
                            <i class="fas fa-filter mr-2"></i>Terapkan
                        </button>
                    </div>
                    <div>
                        <label class="block text-sm font-medium text-gray-700 mb-2">&nbsp;</label>
                        <a href="export_shu_defmart.php?tahun=<?= $filter_tahun_shu ?>" 
                           class="block w-full text-center px-4 py-2 bg-green-600 hover:bg-green-700 text-white rounded-lg font-semibold transition">
                            <i class="fas fa-file-excel mr-2"></i>Export Excel
                        </a>
                    </div>
                </div>
            </div>

            <!-- Summary Cards -->
            <div class="grid grid-cols-1 md:grid-cols-3 gap-6 mb-6">
                <div class="bg-white rounded-xl shadow-sm p-6 border-l-4 border-purple-500">
                    <div class="flex items-center justify-between">
                        <div>
                            <p class="text-sm font-medium text-gray-600 mb-1">Total Transaksi</p>
                            <p class="text-2xl font-bold text-gray-900">Rp <?= number_format($total_data['grand_total_transaksi'], 0, ',', '.') ?></p>
                        </div>
                        <div class="p-4 gradient-primary rounded-lg">
                            <i class="fas fa-money-bill-wave text-2xl text-white"></i>
                        </div>
                    </div>
                </div>

                <div class="bg-white rounded-xl shadow-sm p-6 border-l-4 border-green-500">
                    <div class="flex items-center justify-between">
                        <div>
                            <p class="text-sm font-medium text-gray-600 mb-1">Jumlah Transaksi</p>
                            <p class="text-2xl font-bold text-gray-900"><?= number_format($total_data['grand_jumlah_transaksi'], 0, ',', '.') ?></p>
                        </div>
                        <div class="p-4 gradient-success rounded-lg">
                            <i class="fas fa-shopping-cart text-2xl text-white"></i>
                        </div>
                    </div>
                </div>

                <div class="bg-white rounded-xl shadow-sm p-6 border-l-4 border-blue-500">
                    <div class="flex items-center justify-between">
                        <div>
                            <p class="text-sm font-medium text-gray-600 mb-1">Anggota Aktif</p>
                            <p class="text-2xl font-bold text-gray-900"><?= number_format($total_data['total_anggota_aktif'], 0, ',', '.') ?></p>
                        </div>
                        <div class="p-4 gradient-info rounded-lg">
                            <i class="fas fa-users text-2xl text-white"></i>
                        </div>
                    </div>
                </div>
            </div>

            <!-- Tabel Laporan SHU -->
            <div class="bg-white rounded-xl shadow-sm overflow-hidden">
                <div class="p-6 border-b border-gray-200">
                    <h3 class="text-lg font-semibold text-gray-800">
                        <i class="fas fa-chart-pie mr-2 text-purple-500"></i>Laporan Agregat SHU Defmart Tahun <?= $filter_tahun_shu ?>
                    </h3>
                    <p class="text-sm text-gray-600 mt-2">Data transaksi anggota di Defmart yang akan menjadi dasar perhitungan SHU</p>
                </div>
                <div class="overflow-x-auto">
                    <table class="w-full">
                        <thead>
                            <tr class="bg-gray-50 border-b border-gray-200">
                                <th class="px-6 py-3 text-left text-sm font-semibold text-gray-700">No</th>
                                <th class="px-6 py-3 text-left text-sm font-semibold text-gray-700">NIP</th>
                                <th class="px-6 py-3 text-left text-sm font-semibold text-gray-700">Nama Anggota</th>
                                <th class="px-6 py-3 text-left text-sm font-semibold text-gray-700">Jumlah Transaksi</th>
                                <th class="px-6 py-3 text-left text-sm font-semibold text-gray-700">Total Pembelian</th>
                                <th class="px-6 py-3 text-left text-sm font-semibold text-gray-700">Persentase</th>
                                <th class="px-6 py-3 text-left text-sm font-semibold text-gray-700">SHU Toko</th>
                                <th class="px-6 py-3 text-left text-sm font-semibold text-gray-700">Total SHU</th>
                            </tr>
                        </thead>
                        <tbody class="divide-y divide-gray-200">
                            <?php 
                            $no = 1;
                            while ($shu = $shu_result->fetch_assoc()): 
                                $persentase = $total_data['grand_total_transaksi'] > 0 ? 
                                    ($shu['total_transaksi'] / $total_data['grand_total_transaksi']) * 100 : 0;
                            ?>
                                <tr class="hover:bg-gray-50 transition">
                                    <td class="px-6 py-4 text-sm font-semibold text-gray-900"><?= $no++ ?></td>
                                    <td class="px-6 py-4 text-sm text-gray-600"><?= htmlspecialchars($shu['nip']) ?></td>
                                    <td class="px-6 py-4 text-sm font-medium text-gray-900"><?= htmlspecialchars($shu['nama']) ?></td>
                                    <td class="px-6 py-4 text-sm text-center">
                                        <span class="px-3 py-1 bg-blue-100 text-blue-800 rounded-full font-semibold">
                                            <?= number_format($shu['jumlah_transaksi'], 0, ',', '.') ?>
                                        </span>
                                    </td>
                                    <td class="px-6 py-4 text-sm font-semibold text-gray-900">
                                        Rp <?= number_format($shu['total_transaksi'], 0, ',', '.') ?>
                                    </td>
                                    <td class="px-6 py-4 text-sm">
                                        <div class="flex items-center">
                                            <div class="w-full bg-gray-200 rounded-full h-2 mr-2">
                                                <div class="gradient-primary h-2 rounded-full" style="width: <?= number_format($persentase, 2) ?>%"></div>
                                            </div>
                                            <span class="font-semibold text-purple-600"><?= number_format($persentase, 2) ?>%</span>
                                        </div>
                                    </td>
                                    <td class="px-6 py-4 text-sm">
                                        <?php if ($shu['shu_toko_tahun_ini'] > 0): ?>
                                            <span class="px-3 py-1 bg-green-100 text-green-800 rounded-lg font-semibold">
                                                Rp <?= number_format($shu['shu_toko_tahun_ini'], 0, ',', '.') ?>
                                            </span>
                                        <?php else: ?>
                                            <span class="text-gray-400 text-xs">Belum dihitung</span>
                                        <?php endif; ?>
                                    </td>
                                    <td class="px-6 py-4 text-sm">
                                        <?php if ($shu['total_shu_tahun_ini'] > 0): ?>
                                            <span class="px-3 py-1 gradient-primary text-white rounded-lg font-semibold">
                                                Rp <?= number_format($shu['total_shu_tahun_ini'], 0, ',', '.') ?>
                                            </span>
                                        <?php else: ?>
                                            <span class="text-gray-400 text-xs">Belum dihitung</span>
                                        <?php endif; ?>
                                    </td>
                                </tr>
                            <?php endwhile; ?>
                        </tbody>
                    </table>
                    <?php if ($shu_result->num_rows == 0): ?>
                        <div class="p-6 text-center text-gray-500">
                            <i class="fas fa-inbox text-4xl mb-3 block text-gray-300"></i>
                            Tidak ada data transaksi untuk tahun <?= $filter_tahun_shu ?>
                        </div>
                    <?php endif; ?>
                </div>
            </div>

            <!-- Keterangan -->
            <div class="bg-blue-50 border border-blue-200 rounded-xl p-6 mt-6">
                <h4 class="font-semibold text-blue-900 mb-3 flex items-center">
                    <i class="fas fa-info-circle mr-2"></i>Keterangan Laporan SHU
                </h4>
                <ul class="space-y-2 text-sm text-blue-800">
                    <li class="flex items-start">
                        <i class="fas fa-check-circle text-blue-600 mt-1 mr-2"></i>
                        <span><strong>Total Pembelian:</strong> Total nilai transaksi anggota di Defmart selama tahun <?= $filter_tahun_shu ?></span>
                    </li>
                    <li class="flex items-start">
                        <i class="fas fa-check-circle text-blue-600 mt-1 mr-2"></i>
                        <span><strong>Persentase:</strong> Kontribusi transaksi anggota terhadap total keseluruhan transaksi Defmart</span>
                    </li>
                    <li class="flex items-start">
                        <i class="fas fa-check-circle text-blue-600 mt-1 mr-2"></i>
                        <span><strong>SHU Toko:</strong> Bagian SHU dari unit Defmart yang sudah dihitung dan dibagikan</span>
                    </li>
                    <li class="flex items-start">
                        <i class="fas fa-check-circle text-blue-600 mt-1 mr-2"></i>
                        <span><strong>Prinsip:</strong> Semakin banyak transaksi di Defmart, semakin besar SHU yang akan diterima anggota</span>
                    </li>
                </ul>
            </div>
        </div>
    </div>
</div>

<script>
    function applyFilter() {
        const bulan = document.getElementById('filter_bulan').value;
        const anggota = document.getElementById('filter_anggota').value;
        const sortBy = document.getElementById('sort_by').value;
        const urutan = document.getElementById('urutan').value;

        window.location.href = `?filter_bulan=${bulan}&filter_anggota=${anggota}&sort_by=${sortBy}&urutan=${urutan}`;
    }

    function applySHUFilter() {
        const tahun = document.getElementById('filter_tahun_shu').value;
        window.location.href = `?filter_tahun_shu=${tahun}`;
    }

    function showTab(tabName) {
        // Hide all tab contents
        document.querySelectorAll('.tab-content').forEach(content => {
            content.classList.add('hidden');
        });
        
        // Remove active class from all tabs
        document.querySelectorAll('.tab-button').forEach(button => {
            button.classList.remove('active');
        });
        
        // Show selected tab content
        document.getElementById('content-' + tabName).classList.remove('hidden');
        
        // Add active class to selected tab
        document.getElementById('tab-' + tabName).classList.add('active');
        
        // Update URL without reload
        const url = new URL(window.location);
        url.searchParams.set('tab', tabName);
        window.history.pushState({}, '', url);
    }

    // Check URL parameter on page load
    window.addEventListener('DOMContentLoaded', function() {
        const urlParams = new URLSearchParams(window.location.search);
        const tab = urlParams.get('tab');
        if (tab === 'shu') {
            showTab('shu');
        }
    });
</script>

</body>
</html>