<?php
session_start();
include 'config.php';

// Pastikan role adalah anggota
if (!isset($_SESSION['role']) || $_SESSION['role'] != 'Anggota') {
    header("Location: index.php");
    exit;
}

$id_anggota = $_SESSION['id_anggota'];

// Aktifkan error mysqli agar mudah debugging
mysqli_report(MYSQLI_REPORT_ERROR | MYSQLI_REPORT_STRICT);

// Query untuk mendapatkan data anggota
$anggotaQuery = "SELECT nama, nip, jabatan FROM anggota WHERE id_anggota = ?";
$stmt = $conn->prepare($anggotaQuery);
$stmt->bind_param("i", $id_anggota);
$stmt->execute();
$anggotaData = $stmt->get_result()->fetch_assoc();

// Query untuk mendapatkan data pinjaman
$pinjamanQuery = "SELECT 
    p.id_pinjaman,
    p.jumlah_pinjaman,
    p.bunga_persen,
    p.total_bayar,
    p.lama_bayar,
    p.angsuran_perbulan,
    p.status,
    p.tanggal_pengajuan,
    p.approved_by_kepalaunit,
    p.tanggal_approve_kepalaunit,
    p.catatan_kepalaunit,
    p.dokumen_pendukung,
    p.persentase_bulanan,
    a.nama as approved_by_nama,
    a.nip as approved_by_nip,
    a.jabatan as approved_by_jabatan
FROM pinjaman p
LEFT JOIN anggota a ON p.approved_by_kepalaunit = a.id_anggota
WHERE p.id_anggota = ?
ORDER BY p.tanggal_pengajuan DESC";

$stmt = $conn->prepare($pinjamanQuery);
$stmt->bind_param("i", $id_anggota);
$stmt->execute();
$pinjamanResult = $stmt->get_result();

// Hitung statistik
$totalPengajuan = 0;
$totalDisetujui = 0;
$totalPending = 0;
$totalDitolak = 0;
$nominalDisetujui = 0;

$pinjamanData = [];
while ($row = $pinjamanResult->fetch_assoc()) {
    $pinjamanData[] = $row;
    $totalPengajuan++;
    
    if ($row['status'] == 'disetujui') {
        $totalDisetujui++;
        $nominalDisetujui += $row['jumlah_pinjaman'];
    } elseif ($row['status'] == 'pending') {
        $totalPending++;
    } elseif ($row['status'] == 'ditolak') {
        $totalDitolak++;
    }
}
?>

<!DOCTYPE html>
<html lang="id">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Status Pinjaman - Koperasi Pusdatin</title>
    <script src="https://cdn.tailwindcss.com"></script>
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.0.0/css/all.min.css">
    <script src="https://cdnjs.cloudflare.com/ajax/libs/jspdf/2.5.1/jspdf.umd.min.js"></script>
    <script>
        tailwind.config = {
            theme: {
                extend: {
                    colors: {
                        primary: '#667eea',
                        secondary: '#764ba2',
                    }
                }
            }
        }
    </script>
    <style>
        @import url('https://fonts.googleapis.com/css2?family=Inter:wght@300;400;500;600;700;800&display=swap');
        body {
            font-family: 'Inter', sans-serif;
        }
    </style>
</head>
<body class="bg-gradient-to-br from-purple-50 to-blue-50">
    
    <!-- Sidebar -->
    <aside id="sidebar" class="fixed top-0 left-0 z-40 w-64 h-screen transition-transform -translate-x-full sm:translate-x-0">
        <div class="h-full px-3 py-4 overflow-y-auto bg-gradient-to-b from-purple-600 to-indigo-700 shadow-2xl">
            <!-- Logo Section -->
            <div class="flex flex-col items-center mb-8 pb-6 border-b border-white/20">
                <div class="w-24 h-24 bg-white rounded-full flex items-center justify-center mb-3 shadow-lg">
                    <img src="img/logo.png" alt="Logo Koperasi" class="w-20 h-20 object-contain" onerror="this.style.display='none'; this.nextElementSibling.style.display='block';">
                    <i class="fas fa-university text-purple-600 text-4xl hidden"></i>
                </div>
                <h2 class="text-white font-bold text-lg text-center">Koperasi Pusdatin</h2>
                <p class="text-purple-200 text-xs mt-1">Sistem Informasi Koperasi</p>
            </div>

            <!-- Profile Section -->
            <div class="mb-6 p-4 bg-white/10 rounded-lg backdrop-blur-sm">
                <div class="flex items-center space-x-3">
                    <div class="w-12 h-12 bg-white rounded-full flex items-center justify-center">
                        <i class="fas fa-user text-purple-600 text-xl"></i>
                    </div>
                    <div class="flex-1 min-w-0">
                        <p class="text-white font-semibold text-sm truncate"><?= htmlspecialchars($anggotaData['nama'] ?? 'Anggota') ?></p>
                        <p class="text-purple-200 text-xs truncate"><?= htmlspecialchars($anggotaData['nip'] ?? '-') ?></p>
                    </div>
                </div>
            </div>

            <!-- Navigation Menu -->
            <nav class="space-y-2">
                <a href="dashboard_anggota.php" class="flex items-center p-3 text-white rounded-lg hover:bg-white/20 transition-all duration-200 group">
                    <i class="fas fa-home w-6 text-center group-hover:scale-110 transition-transform"></i>
                    <span class="ml-3 font-medium">Dashboard</span>
                </a>
                
                <a href="ajukan_pinjaman.php" class="flex items-center p-3 text-white rounded-lg hover:bg-white/20 transition-all duration-200 group">
                    <i class="fas fa-money-bill-wave w-6 text-center group-hover:scale-110 transition-transform"></i>
                    <span class="ml-3 font-medium">Ajukan Pinjaman</span>
                </a>
                
                <a href="status_pinjaman.php" class="flex items-center p-3 text-white bg-white/20 rounded-lg hover:bg-white/30 transition-all duration-200 group">
                    <i class="fas fa-file-contract w-6 text-center group-hover:scale-110 transition-transform"></i>
                    <span class="ml-3 font-medium">Status Pinjaman</span>
                </a>
                
                <a href="lihat_transaksi_defmart.php" class="flex items-center p-3 text-white rounded-lg hover:bg-white/20 transition-all duration-200 group">
                    <i class="fas fa-shopping-cart w-6 text-center group-hover:scale-110 transition-transform"></i>
                    <span class="ml-3 font-medium">Transaksi Defmart</span>
                </a>
                
                <a href="lihat_shu.php" class="flex items-center p-3 text-white rounded-lg hover:bg-white/20 transition-all duration-200 group">
                    <i class="fas fa-chart-line w-6 text-center group-hover:scale-110 transition-transform"></i>
                    <span class="ml-3 font-medium">Pembagian SHU</span>
                </a>

                <div class="pt-4 mt-4 border-t border-white/20">
                    <a href="logout.php" class="flex items-center p-3 text-white rounded-lg hover:bg-red-500/30 transition-all duration-200 group">
                        <i class="fas fa-sign-out-alt w-6 text-center group-hover:scale-110 transition-transform"></i>
                        <span class="ml-3 font-medium">Logout</span>
                    </a>
                </div>
            </nav>
        </div>
    </aside>

    <!-- Main Content -->
    <div class="sm:ml-64">
        <!-- Top Navbar -->
        <nav class="bg-white shadow-md sticky top-0 z-30">
            <div class="px-4 py-3 lg:px-6">
                <div class="flex items-center justify-between">
                    <button id="toggleSidebar" class="inline-flex items-center p-2 text-sm text-gray-500 rounded-lg sm:hidden hover:bg-gray-100 focus:outline-none focus:ring-2 focus:ring-gray-200">
                        <i class="fas fa-bars text-xl"></i>
                    </button>
                    <h1 class="text-xl font-bold text-gray-800">Status Pinjaman</h1>
                    <div class="flex items-center space-x-4">
                        <span class="text-sm text-gray-600 hidden md:block">
                            <i class="far fa-calendar-alt mr-2"></i>
                            <?= date('d F Y') ?>
                        </span>
                    </div>
                </div>
            </div>
        </nav>

        <!-- Main Content Area -->
        <div class="p-4 lg:p-6">
            <!-- Statistics Cards -->
            <div class="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-4 gap-6 mb-6">
                <!-- Total Pengajuan -->
                <div class="bg-white rounded-xl shadow-lg hover:shadow-xl transition-all duration-300 overflow-hidden group">
                    <div class="p-6">
                        <div class="flex items-center justify-between mb-4">
                            <div class="p-3 bg-blue-100 rounded-lg group-hover:scale-110 transition-transform">
                                <i class="fas fa-file-alt text-blue-600 text-2xl"></i>
                            </div>
                        </div>
                        <h3 class="text-gray-500 text-sm font-medium mb-2">Total Pengajuan</h3>
                        <p class="text-3xl font-bold text-gray-800"><?= $totalPengajuan ?></p>
                    </div>
                    <div class="bg-blue-50 px-6 py-3">
                        <p class="text-xs text-blue-700">
                            <i class="fas fa-list mr-1"></i>
                            Semua pengajuan
                        </p>
                    </div>
                </div>

                <!-- Disetujui -->
                <div class="bg-white rounded-xl shadow-lg hover:shadow-xl transition-all duration-300 overflow-hidden group">
                    <div class="p-6">
                        <div class="flex items-center justify-between mb-4">
                            <div class="p-3 bg-green-100 rounded-lg group-hover:scale-110 transition-transform">
                                <i class="fas fa-check-circle text-green-600 text-2xl"></i>
                            </div>
                        </div>
                        <h3 class="text-gray-500 text-sm font-medium mb-2">Disetujui</h3>
                        <p class="text-3xl font-bold text-gray-800"><?= $totalDisetujui ?></p>
                    </div>
                    <div class="bg-green-50 px-6 py-3">
                        <p class="text-xs text-green-700">
                            <i class="fas fa-money-bill-wave mr-1"></i>
                            Rp <?= number_format($nominalDisetujui, 0, ',', '.') ?>
                        </p>
                    </div>
                </div>

                <!-- Pending -->
                <div class="bg-white rounded-xl shadow-lg hover:shadow-xl transition-all duration-300 overflow-hidden group">
                    <div class="p-6">
                        <div class="flex items-center justify-between mb-4">
                            <div class="p-3 bg-yellow-100 rounded-lg group-hover:scale-110 transition-transform">
                                <i class="fas fa-clock text-yellow-600 text-2xl"></i>
                            </div>
                        </div>
                        <h3 class="text-gray-500 text-sm font-medium mb-2">Pending</h3>
                        <p class="text-3xl font-bold text-gray-800"><?= $totalPending ?></p>
                    </div>
                    <div class="bg-yellow-50 px-6 py-3">
                        <p class="text-xs text-yellow-700">
                            <i class="fas fa-hourglass-half mr-1"></i>
                            Menunggu persetujuan
                        </p>
                    </div>
                </div>

                <!-- Ditolak -->
                <div class="bg-white rounded-xl shadow-lg hover:shadow-xl transition-all duration-300 overflow-hidden group">
                    <div class="p-6">
                        <div class="flex items-center justify-between mb-4">
                            <div class="p-3 bg-red-100 rounded-lg group-hover:scale-110 transition-transform">
                                <i class="fas fa-times-circle text-red-600 text-2xl"></i>
                            </div>
                        </div>
                        <h3 class="text-gray-500 text-sm font-medium mb-2">Ditolak</h3>
                        <p class="text-3xl font-bold text-gray-800"><?= $totalDitolak ?></p>
                    </div>
                    <div class="bg-red-50 px-6 py-3">
                        <p class="text-xs text-red-700">
                            <i class="fas fa-ban mr-1"></i>
                            Tidak disetujui
                        </p>
                    </div>
                </div>
            </div>

            <!-- Data Pinjaman -->
            <div class="bg-white rounded-xl shadow-lg overflow-hidden">
                <div class="p-6 border-b border-gray-200">
                    <h3 class="text-lg font-bold text-gray-800 flex items-center">
                        <i class="fas fa-list-alt text-purple-600 mr-2"></i>
                        Riwayat Pengajuan Pinjaman
                    </h3>
                </div>

                <?php if (empty($pinjamanData)): ?>
                    <div class="p-12 text-center">
                        <i class="fas fa-inbox text-gray-300 text-6xl mb-4"></i>
                        <p class="text-gray-500 text-lg font-medium">Belum ada pengajuan pinjaman</p>
                        <p class="text-gray-400 text-sm mt-2">Anda belum pernah mengajukan pinjaman</p>
                        <a href="ajukan_pinjaman.php" class="inline-block mt-6 px-6 py-3 bg-purple-600 text-white rounded-lg hover:bg-purple-700 transition-colors">
                            <i class="fas fa-plus mr-2"></i>
                            Ajukan Pinjaman Baru
                        </a>
                    </div>
                <?php else: ?>
                    <div class="overflow-x-auto">
                        <table class="w-full">
                            <thead class="bg-gray-50">
                                <tr>
                                    <th class="px-6 py-4 text-left text-xs font-semibold text-gray-600 uppercase tracking-wider">No</th>
                                    <th class="px-6 py-4 text-left text-xs font-semibold text-gray-600 uppercase tracking-wider">Tanggal Pengajuan</th>
                                    <th class="px-6 py-4 text-left text-xs font-semibold text-gray-600 uppercase tracking-wider">Jumlah Pinjaman</th>
                                    <th class="px-6 py-4 text-left text-xs font-semibold text-gray-600 uppercase tracking-wider">Bunga (%)</th>
                                    <th class="px-6 py-4 text-left text-xs font-semibold text-gray-600 uppercase tracking-wider">Persentase Bulanan</th>
                                    <th class="px-6 py-4 text-left text-xs font-semibold text-gray-600 uppercase tracking-wider">Tenor</th>
                                    <th class="px-6 py-4 text-left text-xs font-semibold text-gray-600 uppercase tracking-wider">Total Bayar</th>
                                    <th class="px-6 py-4 text-left text-xs font-semibold text-gray-600 uppercase tracking-wider">Angsuran/Bulan</th>
                                    <th class="px-6 py-4 text-left text-xs font-semibold text-gray-600 uppercase tracking-wider">Status</th>
                                    <th class="px-6 py-4 text-left text-xs font-semibold text-gray-600 uppercase tracking-wider">Aksi</th>
                                </tr>
                            </thead>
                            <tbody class="divide-y divide-gray-200">
                                <?php foreach ($pinjamanData as $index => $pinjaman): ?>
                                    <tr class="hover:bg-gray-50 transition-colors">
                                        <td class="px-6 py-4 whitespace-nowrap text-sm text-gray-800 font-medium"><?= $index + 1 ?></td>
                                        <td class="px-6 py-4 whitespace-nowrap text-sm text-gray-600">
                                            <i class="far fa-calendar mr-1"></i>
                                            <?= date('d/m/Y', strtotime($pinjaman['tanggal_pengajuan'])) ?>
                                        </td>
                                        <td class="px-6 py-4 whitespace-nowrap text-sm font-semibold text-gray-800">
                                            Rp <?= number_format($pinjaman['jumlah_pinjaman'], 0, ',', '.') ?>
                                        </td>
                                        <td class="px-6 py-4 whitespace-nowrap text-sm text-gray-600">
                                            <?= number_format($pinjaman['bunga_persen'], 2) ?>%
                                        </td>
                                        <td class="px-6 py-4 whitespace-nowrap text-sm text-gray-600">
                                            <?= number_format($pinjaman['persentase_bulanan'], 2) ?>%
                                        </td>
                                        <td class="px-6 py-4 whitespace-nowrap text-sm text-gray-600">
                                            <?= $pinjaman['lama_bayar'] ?> Bulan
                                        </td>
                                        <td class="px-6 py-4 whitespace-nowrap text-sm font-semibold text-gray-800">
                                            <?php if ($pinjaman['total_bayar']): ?>
                                                Rp <?= number_format($pinjaman['total_bayar'], 0, ',', '.') ?>
                                            <?php else: ?>
                                                <span class="text-gray-400">-</span>
                                            <?php endif; ?>
                                        </td>
                                        <td class="px-6 py-4 whitespace-nowrap text-sm text-gray-600">
                                            <?php if ($pinjaman['angsuran_perbulan']): ?>
                                                Rp <?= number_format($pinjaman['angsuran_perbulan'], 0, ',', '.') ?>
                                            <?php else: ?>
                                                <span class="text-gray-400">-</span>
                                            <?php endif; ?>
                                        </td>
                                        <td class="px-6 py-4 whitespace-nowrap">
                                            <?php
                                            $statusClass = '';
                                            $statusIcon = '';
                                            $statusText = '';
                                            
                                            switch ($pinjaman['status']) {
                                                case 'disetujui':
                                                    $statusClass = 'bg-green-100 text-green-700';
                                                    $statusIcon = 'fa-check-circle';
                                                    $statusText = 'Disetujui';
                                                    break;
                                                case 'pending':
                                                    $statusClass = 'bg-yellow-100 text-yellow-700';
                                                    $statusIcon = 'fa-clock';
                                                    $statusText = 'Pending';
                                                    break;
                                                case 'ditolak':
                                                    $statusClass = 'bg-red-100 text-red-700';
                                                    $statusIcon = 'fa-times-circle';
                                                    $statusText = 'Ditolak';
                                                    break;
                                            }
                                            ?>
                                            <span class="px-3 py-1 inline-flex text-xs leading-5 font-semibold rounded-full <?= $statusClass ?>">
                                                <i class="fas <?= $statusIcon ?> mr-1"></i>
                                                <?= $statusText ?>
                                            </span>
                                        </td>
                                        <td class="px-6 py-4 whitespace-nowrap text-sm space-x-2">
                                            <button onclick="showDetail(<?= htmlspecialchars(json_encode($pinjaman)) ?>)" 
                                                    class="text-purple-600 hover:text-purple-900 font-medium">
                                                <i class="fas fa-eye mr-1"></i>
                                                Detail
                                            </button>
                                            <?php if ($pinjaman['status'] != 'pending'): ?>
                                                <button onclick="downloadPDF(<?= htmlspecialchars(json_encode($pinjaman)) ?>, '<?= htmlspecialchars($anggotaData['nama']) ?>', '<?= htmlspecialchars($anggotaData['nip']) ?>')" 
                                                        class="text-blue-600 hover:text-blue-900 font-medium">
                                                    <i class="fas fa-file-pdf mr-1"></i>
                                                    PDF
                                                </button>
                                            <?php endif; ?>
                                        </td>
                                    </tr>
                                <?php endforeach; ?>
                            </tbody>
                        </table>
                    </div>
                <?php endif; ?>
            </div>

            <!-- Footer -->
            <div class="mt-8 text-center text-gray-500 text-sm">
                <p>&copy; 2025 Koperasi Pusdatin. All rights reserved.</p>
            </div>
        </div>
    </div>

    <!-- Modal Detail -->
    <div id="detailModal" class="fixed inset-0 bg-gray-900 bg-opacity-50 hidden z-50 flex items-center justify-center p-4">
        <div class="bg-white rounded-2xl shadow-2xl max-w-3xl w-full max-h-[90vh] overflow-y-auto">
            <div class="p-6 border-b border-gray-200 bg-gradient-to-r from-purple-600 to-indigo-600">
                <div class="flex items-center justify-between">
                    <h3 class="text-xl font-bold text-white flex items-center">
                        <i class="fas fa-info-circle mr-2"></i>
                        Detail Pinjaman
                    </h3>
                    <button onclick="closeModal()" class="text-white hover:text-gray-200 transition-colors">
                        <i class="fas fa-times text-2xl"></i>
                    </button>
                </div>
            </div>
            
            <div id="modalContent" class="p-6">
                <!-- Content will be inserted here -->
            </div>
        </div>
    </div>

    <script>
        // Toggle sidebar untuk mobile
        const sidebar = document.getElementById('sidebar');
        const toggleButton = document.getElementById('toggleSidebar');
        
        toggleButton.addEventListener('click', () => {
            sidebar.classList.toggle('-translate-x-full');
        });

        // Close sidebar when clicking outside on mobile
        document.addEventListener('click', (e) => {
            if (window.innerWidth < 640) {
                if (!sidebar.contains(e.target) && !toggleButton.contains(e.target)) {
                    sidebar.classList.add('-translate-x-full');
                }
            }
        });

        // Modal functions
        function showDetail(data) {
            const modal = document.getElementById('detailModal');
            const content = document.getElementById('modalContent');
            
            let statusBadge = '';
            if (data.status === 'disetujui') {
                statusBadge = '<span class="px-4 py-2 bg-green-100 text-green-700 rounded-full text-sm font-semibold"><i class="fas fa-check-circle mr-1"></i>Disetujui</span>';
            } else if (data.status === 'pending') {
                statusBadge = '<span class="px-4 py-2 bg-yellow-100 text-yellow-700 rounded-full text-sm font-semibold"><i class="fas fa-clock mr-1"></i>Pending</span>';
            } else if (data.status === 'ditolak') {
                statusBadge = '<span class="px-4 py-2 bg-red-100 text-red-700 rounded-full text-sm font-semibold"><i class="fas fa-times-circle mr-1"></i>Ditolak</span>';
            }
            
            let approvalSection = '';
            if (data.status === 'disetujui' && data.tanggal_approve_kepalaunit) {
                const approveDate = new Date(data.tanggal_approve_kepalaunit);
                approvalSection = `
                    <div class="mt-6 p-6 bg-green-50 rounded-xl border-2 border-green-200">
                        <h4 class="font-bold text-green-800 mb-4 flex items-center text-lg">
                            <i class="fas fa-check-circle mr-2"></i>
                            Informasi Persetujuan
                        </h4>
                        <div class="space-y-3">
                            <div class="flex items-start">
                                <i class="fas fa-user-check text-green-600 mt-1 mr-3"></i>
                                <div>
                                    <p class="text-xs text-gray-500 mb-1">Disetujui oleh</p>
                                    <p class="font-semibold text-gray-800">${data.approved_by_nama || '-'}</p>
                                    ${data.approved_by_nip ? `<p class="text-sm text-gray-600">NIP: ${data.approved_by_nip}</p>` : ''}
                                    ${data.approved_by_jabatan ? `<p class="text-sm text-gray-600">${data.approved_by_jabatan}</p>` : ''}
                                </div>
                            </div>
                            <div class="flex items-start">
                                <i class="fas fa-calendar-check text-green-600 mt-1 mr-3"></i>
                                <div>
                                    <p class="text-xs text-gray-500 mb-1">Tanggal Persetujuan</p>
                                    <p class="font-semibold text-gray-800">${approveDate.toLocaleDateString('id-ID', { weekday: 'long', year: 'numeric', month: 'long', day: 'numeric' })}</p>
                                </div>
                            </div>
                            ${data.catatan_kepalaunit ? `
                            <div class="flex items-start">
                                <i class="fas fa-sticky-note text-green-600 mt-1 mr-3"></i>
                                <div>
                                    <p class="text-xs text-gray-500 mb-1">Catatan</p>
                                    <p class="text-gray-700">${data.catatan_kepalaunit}</p>
                                </div>
                            </div>
                            ` : ''}
                        </div>
                    </div>
                `;
            } else if (data.status === 'ditolak' && data.tanggal_approve_kepalaunit) {
                const rejectDate = new Date(data.tanggal_approve_kepalaunit);
                approvalSection = `
                    <div class="mt-6 p-6 bg-red-50 rounded-xl border-2 border-red-200">
                        <h4 class="font-bold text-red-800 mb-4 flex items-center text-lg">
                            <i class="fas fa-times-circle mr-2"></i>
                            Informasi Penolakan
                        </h4>
                        <div class="space-y-3">
                            <div class="flex items-start">
                                <i class="fas fa-user-times text-red-600 mt-1 mr-3"></i>
                                <div>
                                    <p class="text-xs text-gray-500 mb-1">Ditolak oleh</p>
                                    <p class="font-semibold text-gray-800">${data.approved_by_nama || '-'}</p>
                                    ${data.approved_by_nip ? `<p class="text-sm text-gray-600">NIP: ${data.approved_by_nip}</p>` : ''}
                                    ${data.approved_by_jabatan ? `<p class="text-sm text-gray-600">${data.approved_by_jabatan}</p>` : ''}
                                </div>
                            </div>
                            <div class="flex items-start">
                                <i class="fas fa-calendar-times text-red-600 mt-1 mr-3"></i>
                                <div>
                                    <p class="text-xs text-gray-500 mb-1">Tanggal Penolakan</p>
                                    <p class="font-semibold text-gray-800">${rejectDate.toLocaleDateString('id-ID', { weekday: 'long', year: 'numeric', month: 'long', day: 'numeric' })}</p>
                                </div>
                            </div>
                            ${data.catatan_kepalaunit ? `
                            <div class="flex items-start">
                                <i class="fas fa-comment-alt text-red-600 mt-1 mr-3"></i>
                                <div>
                                    <p class="text-xs text-gray-500 mb-1">Alasan Penolakan</p>
                                    <p class="text-gray-700">${data.catatan_kepalaunit}</p>
                                </div>
                            </div>
                            ` : ''}
                        </div>
                    </div>
                `;
            }
            
            content.innerHTML = `
                <div class="space-y-6">
                    <div class="flex items-center justify-between pb-4 border-b border-gray-200">
                        <h4 class="text-lg font-semibold text-gray-800">ID Pinjaman: #${data.id_pinjaman}</h4>
                        ${statusBadge}
                    </div>
                    
                    <div class="grid grid-cols-1 md:grid-cols-2 gap-4">
                        <div class="bg-gradient-to-br from-blue-50 to-blue-100 p-4 rounded-lg border border-blue-200">
                            <p class="text-xs text-blue-600 mb-1 font-medium">Tanggal Pengajuan</p>
                            <p class="text-lg font-bold text-gray-800">
                                <i class="far fa-calendar text-blue-600 mr-2"></i>
                                ${new Date(data.tanggal_pengajuan).toLocaleDateString('id-ID')}
                            </p>
                        </div>
                        
                        <div class="bg-gradient-to-br from-green-50 to-green-100 p-4 rounded-lg border border-green-200">
                            <p class="text-xs text-green-600 mb-1 font-medium">Jumlah Pinjaman</p>
                            <p class="text-lg font-bold text-gray-800">
                                <i class="fas fa-money-bill-wave text-green-600 mr-2"></i>
                                Rp ${parseInt(data.jumlah_pinjaman).toLocaleString('id-ID')}
                            </p>
                        </div>
                        
                        <div class="bg-gradient-to-br from-purple-50 to-purple-100 p-4 rounded-lg border border-purple-200">
                            <p class="text-xs text-purple-600 mb-1 font-medium">Bunga (%)</p>
                            <p class="text-lg font-bold text-gray-800">
                                <i class="fas fa-percentage text-purple-600 mr-2"></i>
                                ${parseFloat(data.bunga_persen).toFixed(2)}%
                            </p>
                        </div>

                        <div class="bg-gradient-to-br from-indigo-50 to-indigo-100 p-4 rounded-lg border border-indigo-200">
                            <p class="text-xs text-indigo-600 mb-1 font-medium">Persentase Bulanan</p>
                            <p class="text-lg font-bold text-gray-800">
                                <i class="fas fa-chart-line text-indigo-600 mr-2"></i>
                                ${parseFloat(data.persentase_bulanan).toFixed(2)}%
                            </p>
                        </div>
                        
                        <div class="bg-gradient-to-br from-orange-50 to-orange-100 p-4 rounded-lg border border-orange-200">
                            <p class="text-xs text-orange-600 mb-1 font-medium">Lama Pembayaran</p>
                            <p class="text-lg font-bold text-gray-800">
                                <i class="fas fa-calendar-alt text-orange-600 mr-2"></i>
                                ${data.lama_bayar} Bulan
                            </p>
                        </div>
                        
                        ${data.total_bayar ? `
                        <div class="bg-gradient-to-br from-pink-50 to-pink-100 p-4 rounded-lg border border-pink-200">
                            <p class="text-xs text-pink-600 mb-1 font-medium">Total Pembayaran</p>
                            <p class="text-lg font-bold text-gray-800">
                                <i class="fas fa-receipt text-pink-600 mr-2"></i>
                                Rp ${parseInt(data.total_bayar).toLocaleString('id-ID')}
                            </p>
                        </div>
                        ` : ''}
                        
                        ${data.angsuran_perbulan ? `
                        <div class="bg-gradient-to-br from-cyan-50 to-cyan-100 p-4 rounded-lg border border-cyan-200 md:col-span-2">
                            <p class="text-xs text-cyan-600 mb-1 font-medium">Angsuran per Bulan</p>
                            <p class="text-lg font-bold text-gray-800">
                                <i class="fas fa-coins text-cyan-600 mr-2"></i>
                                Rp ${parseInt(data.angsuran_perbulan).toLocaleString('id-ID')}
                            </p>
                        </div>
                        ` : ''}
                    </div>
                    
                    ${approvalSection}
                </div>
            `;
            
            modal.classList.remove('hidden');
        }
        
        function closeModal() {
            const modal = document.getElementById('detailModal');
            modal.classList.add('hidden');
        }
        
        // Close modal when clicking outside
        document.getElementById('detailModal').addEventListener('click', function(e) {
            if (e.target === this) {
                closeModal();
            }
        });
        
        // Close modal with Escape key
        document.addEventListener('keydown', function(e) {
            if (e.key === 'Escape') {
                closeModal();
            }
        });

        // Function to download PDF
        async function downloadPDF(data, namaPemohon, nipPemohon) {
            const { jsPDF } = window.jspdf;
            const doc = new jsPDF();
            
            // Define colors
            const primaryColor = data.status === 'disetujui' ? [34, 197, 94] : [239, 68, 68];
            const secondaryColor = [102, 126, 234];
            
            // Header with gradient effect
            doc.setFillColor(primaryColor[0], primaryColor[1], primaryColor[2]);
            doc.rect(0, 0, 210, 40, 'F');
            
            // Logo placeholder (using text)
            doc.setFillColor(255, 255, 255);
            doc.circle(20, 20, 10, 'F');
            doc.setTextColor(primaryColor[0], primaryColor[1], primaryColor[2]);
            doc.setFontSize(12);
            doc.setFont(undefined, 'bold');
            doc.text('KP', 15, 23);
            
            // Title
            doc.setTextColor(255, 255, 255);
            doc.setFontSize(20);
            doc.setFont(undefined, 'bold');
            doc.text('KOPERASI PUSDATIN', 105, 20, { align: 'center' });
            doc.setFontSize(12);
            doc.setFont(undefined, 'normal');
            doc.text('Sistem Informasi Koperasi', 105, 28, { align: 'center' });
            
            // Document title
            doc.setFillColor(245, 245, 245);
            doc.rect(0, 45, 210, 15, 'F');
            doc.setTextColor(60, 60, 60);
            doc.setFontSize(16);
            doc.setFont(undefined, 'bold');
            const docTitle = data.status === 'disetujui' ? 'SURAT PERSETUJUAN PINJAMAN' : 'SURAT PENOLAKAN PINJAMAN';
            doc.text(docTitle, 105, 54, { align: 'center' });
            
            // Document number
            doc.setFontSize(10);
            doc.setFont(undefined, 'normal');
            doc.text(`No: ${data.id_pinjaman}/KP-PINJAMAN/${new Date().getFullYear()}`, 105, 68, { align: 'center' });
            
            let yPos = 80;
            
            // Pemohon section
            doc.setFillColor(secondaryColor[0], secondaryColor[1], secondaryColor[2]);
            doc.rect(15, yPos, 180, 8, 'F');
            doc.setTextColor(255, 255, 255);
            doc.setFontSize(11);
            doc.setFont(undefined, 'bold');
            doc.text('DATA PEMOHON', 20, yPos + 5.5);
            
            yPos += 12;
            doc.setTextColor(60, 60, 60);
            doc.setFontSize(10);
            doc.setFont(undefined, 'normal');
            
            const pemohonData = [
                ['Nama', namaPemohon],
                ['NIP', nipPemohon],
                ['Tanggal Pengajuan', new Date(data.tanggal_pengajuan).toLocaleDateString('id-ID', { weekday: 'long', year: 'numeric', month: 'long', day: 'numeric' })]
            ];
            
            pemohonData.forEach(([label, value]) => {
                doc.setFont(undefined, 'bold');
                doc.text(label, 20, yPos);
                doc.setFont(undefined, 'normal');
                doc.text(': ' + value, 70, yPos);
                yPos += 7;
            });
            
            yPos += 5;
            
            // Detail Pinjaman section
            doc.setFillColor(secondaryColor[0], secondaryColor[1], secondaryColor[2]);
            doc.rect(15, yPos, 180, 8, 'F');
            doc.setTextColor(255, 255, 255);
            doc.setFontSize(11);
            doc.setFont(undefined, 'bold');
            doc.text('DETAIL PINJAMAN', 20, yPos + 5.5);
            
            yPos += 12;
            doc.setTextColor(60, 60, 60);
            doc.setFontSize(10);
            doc.setFont(undefined, 'normal');
            
            const detailData = [
                ['Jumlah Pinjaman', 'Rp ' + parseInt(data.jumlah_pinjaman).toLocaleString('id-ID')],
                ['Bunga', parseFloat(data.bunga_persen).toFixed(2) + '%'],
                ['Persentase Bulanan', parseFloat(data.persentase_bulanan).toFixed(2) + '%'],
                ['Lama Pembayaran', data.lama_bayar + ' Bulan']
            ];
            
            if (data.total_bayar) {
                detailData.push(['Total Pembayaran', 'Rp ' + parseInt(data.total_bayar).toLocaleString('id-ID')]);
            }
            
            if (data.angsuran_perbulan) {
                detailData.push(['Angsuran per Bulan', 'Rp ' + parseInt(data.angsuran_perbulan).toLocaleString('id-ID')]);
            }
            
            detailData.forEach(([label, value]) => {
                doc.setFont(undefined, 'bold');
                doc.text(label, 20, yPos);
                doc.setFont(undefined, 'normal');
                doc.text(': ' + value, 70, yPos);
                yPos += 7;
            });
            
            yPos += 5;
            
            // Status section
            const statusColor = data.status === 'disetujui' ? [34, 197, 94] : [239, 68, 68];
            doc.setFillColor(statusColor[0], statusColor[1], statusColor[2]);
            doc.rect(15, yPos, 180, 8, 'F');
            doc.setTextColor(255, 255, 255);
            doc.setFontSize(11);
            doc.setFont(undefined, 'bold');
            const statusTitle = data.status === 'disetujui' ? 'INFORMASI PERSETUJUAN' : 'INFORMASI PENOLAKAN';
            doc.text(statusTitle, 20, yPos + 5.5);
            
            yPos += 12;
            doc.setTextColor(60, 60, 60);
            doc.setFontSize(10);
            doc.setFont(undefined, 'normal');
            
            if (data.approved_by_nama) {
                const statusData = [
                    [data.status === 'disetujui' ? 'Disetujui oleh' : 'Ditolak oleh', data.approved_by_nama],
                ];
                
                if (data.approved_by_nip) {
                    statusData.push(['NIP', data.approved_by_nip]);
                }
                
                if (data.approved_by_jabatan) {
                    statusData.push(['Jabatan', data.approved_by_jabatan]);
                }
                
                if (data.tanggal_approve_kepalaunit) {
                    const approveDate = new Date(data.tanggal_approve_kepalaunit);
                    statusData.push([
                        data.status === 'disetujui' ? 'Tanggal Persetujuan' : 'Tanggal Penolakan',
                        approveDate.toLocaleDateString('id-ID', { weekday: 'long', year: 'numeric', month: 'long', day: 'numeric' })
                    ]);
                }
                
                statusData.forEach(([label, value]) => {
                    doc.setFont(undefined, 'bold');
                    doc.text(label, 20, yPos);
                    doc.setFont(undefined, 'normal');
                    doc.text(': ' + value, 70, yPos);
                    yPos += 7;
                });
                
                if (data.catatan_kepalaunit) {
                    yPos += 2;
                    doc.setFont(undefined, 'bold');
                    doc.text(data.status === 'disetujui' ? 'Catatan:' : 'Alasan Penolakan:', 20, yPos);
                    yPos += 7;
                    doc.setFont(undefined, 'normal');
                    
                    const splitText = doc.splitTextToSize(data.catatan_kepalaunit, 170);
                    doc.text(splitText, 20, yPos);
                    yPos += (splitText.length * 7);
                }
            }
            
            // Footer
            yPos = 270;
            doc.setFillColor(240, 240, 240);
            doc.rect(0, yPos, 210, 27, 'F');
            
            doc.setTextColor(100, 100, 100);
            doc.setFontSize(8);
            doc.setFont(undefined, 'italic');
            doc.text('Dokumen ini dibuat secara otomatis oleh sistem', 105, yPos + 8, { align: 'center' });
            doc.text(`Dicetak pada: ${new Date().toLocaleDateString('id-ID', { weekday: 'long', year: 'numeric', month: 'long', day: 'numeric', hour: '2-digit', minute: '2-digit' })}`, 105, yPos + 13, { align: 'center' });
            
            doc.setFontSize(9);
            doc.setFont(undefined, 'bold');
            doc.text('© 2025 Koperasi Pusdatin - Sistem Informasi Koperasi', 105, yPos + 20, { align: 'center' });
            
            // Save PDF
            const fileName = `Pinjaman_${data.id_pinjaman}_${data.status.toUpperCase()}.pdf`;
            doc.save(fileName);
        }
    </script>
</body>
</html>