<?php
session_start();
include 'config.php';

// Cek apakah user sudah login dan memiliki role Kasir Defmart
if (!isset($_SESSION['role']) || $_SESSION['role'] !== 'Kasir Defmart') {
    header("Location: index.php");
    exit;
}

$id_user = $_SESSION['id_user'];
$username = $_SESSION['username'];

// Proses Tambah Transaksi
if (isset($_POST['tambah_transaksi'])) {
    $id_anggota = trim($_POST['id_anggota']);
    $nama_barang = trim($_POST['nama_barang']);
    $qty = intval($_POST['qty']);
    $harga_satuan = floatval($_POST['harga_satuan']);
    $total_harga = $qty * $harga_satuan;
    $metode_pembayaran = trim($_POST['metode_pembayaran']);

    if ($id_anggota && $nama_barang && $qty > 0 && $harga_satuan > 0) {
        // Mulai transaksi database
        $conn->begin_transaction();
        
        try {
            // Insert ke defmart_transaksi
            $stmt = $conn->prepare("INSERT INTO defmart_transaksi (id_anggota, nama_barang, qty, harga_satuan, total_harga, metode_pembayaran) VALUES (?, ?, ?, ?, ?, ?)");
            if (!$stmt) {
                throw new Exception("Prepare statement gagal: " . $conn->error);
            }
            $stmt->bind_param("isidds", $id_anggota, $nama_barang, $qty, $harga_satuan, $total_harga, $metode_pembayaran);
            $stmt->execute();
            $stmt->close();
            
            // Update total_dana_masuk di tabel unit untuk Defmart
            $stmt_unit = $conn->prepare("UPDATE unit SET total_dana_masuk = total_dana_masuk + ? WHERE nama_unit = 'Defmart'");
            if (!$stmt_unit) {
                throw new Exception("Prepare statement gagal: " . $conn->error);
            }
            $stmt_unit->bind_param("d", $total_harga);
            $stmt_unit->execute();
            $stmt_unit->close();
            
            // Get id_unit untuk Defmart
            $stmt_get_unit = $conn->prepare("SELECT id_unit FROM unit WHERE nama_unit = 'Defmart'");
            if (!$stmt_get_unit) {
                throw new Exception("Prepare statement gagal: " . $conn->error);
            }
            $stmt_get_unit->execute();
            $result_unit = $stmt_get_unit->get_result();
            $unit_data = $result_unit->fetch_assoc();
            
            if (!$unit_data) {
                throw new Exception("Unit Defmart tidak ditemukan");
            }
            
            $id_unit = $unit_data['id_unit'];
            $stmt_get_unit->close();
            
            // Update atau Insert ke tabel dana_unit
            $tahun = date('Y');
            $bulan = date('n'); // 1-12
            
            // Cek apakah sudah ada record untuk tahun dan bulan ini
            $stmt_check = $conn->prepare("SELECT id_dana, total_dana FROM dana_unit WHERE id_unit = ? AND tahun = ? AND bulan = ?");
            if (!$stmt_check) {
                throw new Exception("Prepare statement gagal: " . $conn->error);
            }
            $stmt_check->bind_param("iii", $id_unit, $tahun, $bulan);
            $stmt_check->execute();
            $result_check = $stmt_check->get_result();
            
            if ($result_check->num_rows > 0) {
                // Update existing record
                $stmt_update_dana = $conn->prepare("UPDATE dana_unit SET total_dana = total_dana + ? WHERE id_unit = ? AND tahun = ? AND bulan = ?");
                if (!$stmt_update_dana) {
                    throw new Exception("Prepare statement gagal: " . $conn->error);
                }
                $stmt_update_dana->bind_param("diii", $total_harga, $id_unit, $tahun, $bulan);
                $stmt_update_dana->execute();
                $stmt_update_dana->close();
            } else {
                // Insert new record
                $tanggal_input = date('Y-m-d');
                $nama_bulan = [
                    1 => 'Januari', 2 => 'Februari', 3 => 'Maret', 4 => 'April',
                    5 => 'Mei', 6 => 'Juni', 7 => 'Juli', 8 => 'Agustus',
                    9 => 'September', 10 => 'Oktober', 11 => 'November', 12 => 'Desember'
                ];
                $keterangan = "Dana Defmart bulan " . $nama_bulan[$bulan] . " " . $tahun;
                $stmt_insert_dana = $conn->prepare("INSERT INTO dana_unit (id_unit, tahun, bulan, total_dana, keterangan, tanggal_input) VALUES (?, ?, ?, ?, ?, ?)");
                if (!$stmt_insert_dana) {
                    throw new Exception("Prepare statement gagal: " . $conn->error);
                }
                $stmt_insert_dana->bind_param("iiidss", $id_unit, $tahun, $bulan, $total_harga, $keterangan, $tanggal_input);
                $stmt_insert_dana->execute();
                $stmt_insert_dana->close();
            }
            $stmt_check->close();
            
            // Jika metode pembayaran adalah Bon, masukkan/update ke tabel potongan_gaji
            if ($metode_pembayaran === 'Bon') {
                $bulan_periode = date('Y-m-01'); // Tanggal awal bulan
                
                // Cek apakah sudah ada potongan untuk anggota ini di bulan ini dari Defmart
                $stmt_check_potongan = $conn->prepare("SELECT id_potongan, jumlah FROM potongan_gaji WHERE id_anggota = ? AND sumber = 'Defmart' AND bulan_periode = ? AND status = 'pending'");
                if (!$stmt_check_potongan) {
                    throw new Exception("Prepare statement gagal: " . $conn->error);
                }
                $stmt_check_potongan->bind_param("is", $id_anggota, $bulan_periode);
                $stmt_check_potongan->execute();
                $result_potongan = $stmt_check_potongan->get_result();
                
                if ($result_potongan->num_rows > 0) {
                    // Update jumlah potongan yang sudah ada
                    $potongan_data = $result_potongan->fetch_assoc();
                    $jumlah_baru = $potongan_data['jumlah'] + $total_harga;
                    
                    $stmt_update_potongan = $conn->prepare("UPDATE potongan_gaji SET jumlah = ? WHERE id_potongan = ?");
                    if (!$stmt_update_potongan) {
                        throw new Exception("Prepare statement gagal: " . $conn->error);
                    }
                    $stmt_update_potongan->bind_param("di", $jumlah_baru, $potongan_data['id_potongan']);
                    $stmt_update_potongan->execute();
                    $stmt_update_potongan->close();
                } else {
                    // Insert potongan baru
                    $stmt_insert_potongan = $conn->prepare("INSERT INTO potongan_gaji (id_anggota, sumber, jumlah, bulan_periode, status) VALUES (?, 'Defmart', ?, ?, 'pending')");
                    if (!$stmt_insert_potongan) {
                        throw new Exception("Prepare statement gagal: " . $conn->error);
                    }
                    $stmt_insert_potongan->bind_param("ids", $id_anggota, $total_harga, $bulan_periode);
                    $stmt_insert_potongan->execute();
                    $stmt_insert_potongan->close();
                }
                
                $stmt_check_potongan->close();
            }
            
            // Commit transaksi
            $conn->commit();
            $sukses = "Transaksi berhasil ditambahkan!";
            if ($metode_pembayaran === 'Bon') {
                $sukses .= " Potongan gaji telah dicatat untuk bulan ini.";
            }
            
        } catch (Exception $e) {
            // Rollback jika ada error
            $conn->rollback();
            $error = "Gagal menambahkan transaksi: " . $e->getMessage();
        }
    } else {
        $error = "Semua field harus diisi dengan benar!";
    }
}

// Ambil data anggota untuk dropdown
$anggota_result = $conn->query("SELECT id_anggota, nama FROM anggota ORDER BY nama");
?>

<!DOCTYPE html>
<html lang="id">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Tambah Transaksi - Koperasi PUSDATIN</title>
    <script src="https://cdn.tailwindcss.com"></script>
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.0.0/css/all.min.css">
    <style>
        .gradient-primary { background: linear-gradient(135deg, #667eea 0%, #764ba2 100%); }
    </style>
</head>
<body class="bg-gray-100">

<!-- Sidebar -->
<div class="fixed left-0 top-0 h-screen w-64 gradient-primary text-white z-40">
    <div class="p-6 border-b border-white border-opacity-20">
        <div class="flex items-center space-x-3 mb-4">
            <img src="img/logo.png" alt="Logo" class="w-12 h-12 rounded-full object-cover bg-white p-1" onerror="this.src='data:image/svg+xml,%3Csvg xmlns=%22http://www.w3.org/2000/svg%22 viewBox=%220 0 100 100%22%3E%3Crect fill=%22%23667eea%22 width=%22100%22 height=%22100%22/%3E%3Ctext x=%2250%22 y=%2250%22 font-size=%2240%22 fill=%22white%22 text-anchor=%22middle%22 dy=%22.3em%22%3E%F0%9F%8F%AA%3C/text%3E%3C/svg%3E'">
            <div>
                <h2 class="font-bold text-lg">DEFMART</h2>
                <p class="text-xs text-white text-opacity-75">Kasir</p>
            </div>
        </div>
    </div>

    <nav class="mt-6 px-4 space-y-2">
        <?php
        $current_page = basename($_SERVER['PHP_SELF']);
        ?>
        <a href="kasir_defmart.php" class="nav-item flex items-center space-x-3 px-4 py-3 rounded-lg <?= $current_page == 'kasir_defmart.php' ? 'bg-white bg-opacity-20' : '' ?> text-white transition hover:bg-white hover:bg-opacity-20">
            <i class="fas fa-chart-line w-5"></i>
            <span>Dashboard</span>
        </a>
        <a href="tambah_transaksi_kasir.php" class="nav-item flex items-center space-x-3 px-4 py-3 rounded-lg <?= $current_page == 'tambah_transaksi_kasir.php' ? 'bg-white bg-opacity-20' : '' ?> text-white transition hover:bg-white hover:bg-opacity-20">
            <i class="fas fa-plus-circle w-5"></i>
            <span>Tambah Transaksi</span>
        </a>
        <a href="riwayat_transaksi_kasir.php" class="nav-item flex items-center space-x-3 px-4 py-3 rounded-lg <?= $current_page == 'riwayat_transaksi_kasir.php' ? 'bg-white bg-opacity-20' : '' ?> text-white transition hover:bg-white hover:bg-opacity-20">
            <i class="fas fa-list w-5"></i>
            <span>Riwayat Transaksi</span>
        </a>
        <a href="barang_terlaris_kasir.php" class="nav-item flex items-center space-x-3 px-4 py-3 rounded-lg <?= $current_page == 'barang_terlaris_kasir.php' ? 'bg-white bg-opacity-20' : '' ?> text-white transition hover:bg-white hover:bg-opacity-20">
            <i class="fas fa-fire w-5"></i>
            <span>Barang Terlaris</span>
        </a>
    </nav>

    <div class="absolute bottom-0 left-0 right-0 p-4 border-t border-white border-opacity-20">
        <div class="bg-white bg-opacity-20 rounded-lg p-3 mb-3">
            <p class="text-xs text-white text-opacity-75">Pengguna</p>
            <p class="font-semibold text-sm"><?= htmlspecialchars($_SESSION['username']) ?></p>
        </div>
        <a href="logout.php" class="w-full block text-center px-4 py-2 bg-red-500 hover:bg-red-600 rounded-lg text-white font-semibold transition">
            <i class="fas fa-sign-out-alt me-2"></i>Logout
        </a>
    </div>
</div>

<!-- Main Content -->
<div class="ml-64 min-h-screen">
    <!-- Top Bar -->
    <div class="bg-white shadow-sm sticky top-0 z-30">
        <div class="px-6 py-4 flex justify-between items-center">
            <h1 class="text-2xl font-bold text-gray-800">Tambah Transaksi Baru</h1>
            <a href="kasir_defmart.php" class="text-gray-600 hover:text-gray-800">
                <i class="fas fa-arrow-left mr-2"></i>Kembali ke Dashboard
            </a>
        </div>
    </div>

    <!-- Content Area -->
    <div class="p-6">
        <!-- Alert Messages -->
        <?php if (isset($sukses)): ?>
            <div class="mb-4 p-4 bg-green-50 border border-green-200 rounded-lg flex items-start">
                <i class="fas fa-check-circle text-green-500 mt-1 mr-3"></i>
                <div>
                    <p class="text-green-800 font-semibold"><?= $sukses ?></p>
                    <a href="riwayat_transaksi_kasir.php" class="text-green-700 underline text-sm mt-2 inline-block">Lihat Riwayat Transaksi</a>
                </div>
            </div>
        <?php endif; ?>

        <?php if (isset($error)): ?>
            <div class="mb-4 p-4 bg-red-50 border border-red-200 rounded-lg flex items-start">
                <i class="fas fa-exclamation-circle text-red-500 mt-1 mr-3"></i>
                <div>
                    <p class="text-red-800 font-semibold"><?= $error ?></p>
                </div>
            </div>
        <?php endif; ?>

        <!-- Form Tambah Transaksi -->
        <div class="max-w-4xl mx-auto">
            <div class="bg-white rounded-xl shadow-sm p-8">
                <div class="mb-6">
                    <h3 class="text-xl font-semibold text-gray-800 flex items-center">
                        <i class="fas fa-shopping-cart mr-3 text-blue-500"></i>
                        Input Data Transaksi
                    </h3>
                    <p class="text-sm text-gray-600 mt-2">Lengkapi formulir di bawah ini untuk menambahkan transaksi baru</p>
                </div>

                <form method="POST" class="space-y-6">
                    <!-- Anggota & Barang -->
                    <div class="grid grid-cols-1 md:grid-cols-2 gap-6">
                        <div>
                            <label class="block text-sm font-medium text-gray-700 mb-2">
                                Pilih Anggota <span class="text-red-500">*</span>
                            </label>
                            <select name="id_anggota" class="w-full px-4 py-3 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-transparent" required>
                                <option value="">-- Pilih Anggota --</option>
                                <?php 
                                while ($anggota = $anggota_result->fetch_assoc()): 
                                ?>
                                    <option value="<?= $anggota['id_anggota'] ?>">
                                        <?= htmlspecialchars($anggota['nama']) ?>
                                    </option>
                                <?php endwhile; ?>
                            </select>
                        </div>

                        <div>
                            <label class="block text-sm font-medium text-gray-700 mb-2">
                                Nama Barang <span class="text-red-500">*</span>
                            </label>
                            <input type="text" name="nama_barang" class="w-full px-4 py-3 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-transparent" placeholder="Masukkan nama barang" required>
                        </div>
                    </div>

                    <!-- Qty, Harga, Pembayaran -->
                    <div class="grid grid-cols-1 md:grid-cols-3 gap-6">
                        <div>
                            <label class="block text-sm font-medium text-gray-700 mb-2">
                                Jumlah (Qty) <span class="text-red-500">*</span>
                            </label>
                            <input type="number" name="qty" id="qty" class="w-full px-4 py-3 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-transparent" placeholder="0" min="1" required>
                        </div>

                        <div>
                            <label class="block text-sm font-medium text-gray-700 mb-2">
                                Harga Satuan <span class="text-red-500">*</span>
                            </label>
                            <div class="relative">
                                <span class="absolute left-4 top-3 text-gray-500">Rp</span>
                                <input type="number" name="harga_satuan" id="harga_satuan" class="w-full pl-12 pr-4 py-3 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-transparent" placeholder="0" min="1" step="1" required>
                            </div>
                        </div>

                        <div>
                            <label class="block text-sm font-medium text-gray-700 mb-2">
                                Metode Pembayaran <span class="text-red-500">*</span>
                            </label>
                            <select name="metode_pembayaran" class="w-full px-4 py-3 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-transparent" required>
                                <option value="">-- Pilih Metode --</option>
                                <option value="Cash">💵 Cash</option>
                                <option value="Debit">💳 Debit</option>
                                <option value="Bon">📝 Bon (Potong Gaji)</option>
                            </select>
                        </div>
                    </div>

                    <!-- Total Harga Display -->
                    <div class="bg-gradient-to-r from-blue-50 to-purple-50 rounded-lg p-6">
                        <label class="block text-sm font-medium text-gray-700 mb-2">Total Harga</label>
                        <div class="text-3xl font-bold text-gray-900" id="total_harga_display">Rp 0</div>
                        <p class="text-sm text-gray-600 mt-1">Total otomatis dihitung dari qty × harga satuan</p>
                    </div>

                    <!-- Info Box -->
                    <div class="bg-yellow-50 border-l-4 border-yellow-400 p-4 rounded">
                        <div class="flex items-start">
                            <i class="fas fa-info-circle text-yellow-600 mt-1 mr-3"></i>
                            <div>
                                <p class="text-sm text-yellow-800 font-semibold">Informasi Pembayaran Bon</p>
                                <p class="text-sm text-yellow-700 mt-1">Jika memilih metode pembayaran <strong>Bon</strong>, nilai transaksi akan otomatis dicatat sebagai potongan gaji anggota untuk bulan berjalan. Jika anggota sudah memiliki bon di bulan yang sama, jumlahnya akan dijumlahkan.</p>
                            </div>
                        </div>
                    </div>

                    <!-- Action Buttons -->
                    <div class="flex gap-4 pt-4">
                        <button type="submit" name="tambah_transaksi" class="flex-1 gradient-primary text-white font-semibold py-3 rounded-lg hover:shadow-lg transition">
                            <i class="fas fa-save mr-2"></i>Simpan Transaksi
                        </button>
                        <a href="kasir_defmart.php" class="px-6 py-3 bg-gray-200 hover:bg-gray-300 text-gray-700 font-semibold rounded-lg transition">
                            <i class="fas fa-times mr-2"></i>Batal
                        </a>
                    </div>
                </form>
            </div>
        </div>
    </div>
</div>

<script>
    // Auto calculate total price
    const qtyInput = document.getElementById('qty');
    const hargaInput = document.getElementById('harga_satuan');
    const totalDisplay = document.getElementById('total_harga_display');

    function calculateTotal() {
        const qty = parseFloat(qtyInput.value) || 0;
        const harga = parseFloat(hargaInput.value) || 0;
        const total = qty * harga;
        
        totalDisplay.textContent = 'Rp ' + new Intl.NumberFormat('id-ID').format(total);
    }

    qtyInput.addEventListener('input', calculateTotal);
    hargaInput.addEventListener('input', calculateTotal);
</script>

</body>
</html>